/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

/*
 * qfle3f_procfs.c: QLogic 10 Gigabit ESX Ethernet FCoE offload driver.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation.
 *
 */

#include <vmkapi.h>
#include "qfle3f.h"
#include "qfle3f_version.h"

#define MAX_KEY_VAL_STRING_LEN 4096

extern struct qfle3fDriverInfo_t qfle3fDriverInfo;

/*
 * A default, empty vmk_MgmtApiSignature on which we can attach
 * key-value pairs.
 */
vmk_MgmtApiSignature kvSig = {
	.version = VMK_REVISION_FROM_NUMBERS(1,0,0,0),
	.name.string = "qfle3f",
	.vendor.string = "qlogic",
	.numCallbacks = 0,
	.callbacks = NULL,
};

//#define _QFLE3F_EN_STATS_
enum _qfle3f_verbose_level{
	QFLE3F_VERBOSE_FC = 0,
	QFLE3F_VERBOSE_HBA,
	QFLE3F_VERBOSE_LPORT,
	QFLE3F_VERBOSE_RPORT,

	/* END.. */
	QFLE3F_VERBOSE_ALL,
};

static vmk_uint32 qfle3f_verbose = QFLE3F_VERBOSE_LPORT;


struct qfle3f_proc_param {
	const char *name;
	int type;
	vmk_uint32 *value;
};

#define QFLE3F_PARAM_TYPE_HEX		1
#define QFLE3F_PARAM_TYPE_DEC		2
#define QFLE3F_PARAM_TYPE_STR		3

struct info_str {
	char *buffer;
	int length;
    int finalLength;
};

#define copy_info(info, fmt, ...) \
    { \
	vmk_StringFormat((info)->buffer+(info)->length, (info)->finalLength - (info)->length, NULL, fmt, ##__VA_ARGS__); \
	    (info)->length = vmk_Strnlen((info)->buffer, (info)->finalLength); \
    }

static void
qfle3f_fcinfo_fcf(struct info_str *info, struct qfle3fHBA *hba)
{
	copy_info(info, "   MAC Address         "
				"%02x:%02x:%02x:%02x:%02x:%02x\n",
			hba->grantedMacAddress[0], hba->grantedMacAddress[1], hba->grantedMacAddress[2],
			hba->grantedMacAddress[3], hba->grantedMacAddress[4], hba->grantedMacAddress[5]);
}

static void
qfle3f_fcinfo_wwn(struct info_str *info, struct qfle3fHBA *hba)
{
	copy_info(info, "Physical Port WWNN    %16.16lx\n", qfle3f_get_wwnn(hba));
	copy_info(info, "Physical Port WWPN    %16.16lx\n", qfle3f_get_wwpn(hba));
	copy_info(info, "PortID                0x%x\n",
			hba->initiator_targetPortID);
}

const char *qfle3f_lportState_str[] = {
	"LPORT_ST_DISABLED", "LPORT_ST_FLOGI", "LPORT_ST_DNS",
	"LPORT_ST_RNN_ID", "LPORT_ST_RSNN_NN", "LPORT_ST_RSPN_ID",
	"LPORT_ST_RFT_ID", "LPORT_ST_RFF_ID", "LPORT_ST_SCR",
	"LPORT_ST_READY", "LPORT_ST_LOGO", "LPORT_ST_RESET"
};

static void
qfle3f_show_fcinfo(struct qfle3fHBA *hba, struct info_str *info)
{
	qfle3f_fcinfo_wwn(info, hba);
	qfle3f_fcinfo_fcf(info, hba);
	/* qfle3f_fcinfo_rport(info, hba); */
}

static void
qfle3f_show_cmd(struct info_str *info, struct qfle3fCommand *ioRequest)
{
	struct io_bdt *bdt = NULL;
	vmk_uint16 i = 0;

	if (!info || !ioRequest)
		return;

	copy_info(info, "      qfle3fCommand(%p): type: ", ioRequest);

	switch (ioRequest->cmd_type) {
	case QFLE3F_SCSI_CMD:
		copy_info(info, "SCSI_CMD\n");
		break;
	case QFLE3F_TASK_MGMT_CMD:
		copy_info(info, "TASK_MGMT\n");
		break;
	case QFLE3F_ABTS:
		copy_info(info, "ABTS\n");
		break;
	case QFLE3F_ELS:
		copy_info(info, "ELS\n");
		break;
	case QFLE3F_CLEANUP:
		copy_info(info, "CLEANUP\n");
		break;
	default:
		copy_info(info, "Unknown\n");
		break;
	}

	if (qfle3f_verbose < QFLE3F_VERBOSE_ALL)
		return;

	copy_info(info, "      ioRequest_flags:0x%x target:%p sc_cmd:%p\n",
			ioRequest->ioRequest_flags, ioRequest->target, ioRequest->sc_cmd);

	copy_info(info, "      xid:0x%x data_xfr_len:%ld"
			" fcp_rsp:%p\n", ioRequest->xid,
			ioRequest->data_xfer_len, ioRequest->rsp);

	copy_info(info, "      fcp_resid:0x%d fcp_rsp_len:%d fcp_sns_len:%d\n",
			ioRequest->fcp_resid, ioRequest->fcp_rsp_len, ioRequest->fcp_sns_len);

	bdt = ioRequest->bd_tbl;

	for (i = 0; i < bdt->bd_valid; i++) {
		copy_info(info, "       BD[%i]:addr_hi:lo[0x%x:%x] len:%d "
			"Flags:0x%x\n", i, bdt->bd_tbl->buf_addr_hi,
			bdt->bd_tbl->buf_addr_lo, bdt->bd_tbl->buf_len,
			bdt->bd_tbl->flags);
	}

	copy_info(info, "      req_flags: following bits set - ");
	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_ISSUE_RRQ))
		copy_info(info, "ISSUE_RRQ, ");

	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_ISSUE_ABTS))
		copy_info(info, "ISSUE_ABTS, ");

	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_ABTS_DONE))
		copy_info(info, "ABTS_DONE, ");

	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_TM_COMPL))
		copy_info(info, "TM_COMPL, ");

	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_TM_TIMEOUT))
		copy_info(info, "TM_TIMEOUT, ");

	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_IO_CLEANUP))
		copy_info(info, "IO_CLEANUP, ");

	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_RETIRE_OXID))
		copy_info(info, "RETIRE_OXID, ");

	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_EH_ABORT))
		copy_info(info, "ABORT, ");

	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_IO_COMPL))
		copy_info(info, "IO_COMPL, ");

	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_ELS_DONE))
		copy_info(info, "ELS_DONE, ");

	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_ELS_TIMEOUT))
		copy_info(info, "ELS_TIMEOUT");

	copy_info(info, "\n");

	copy_info(info, "      cdb_status:0x%d fcp_status:%d fcp_rsp_code:%d"
			" scsi_comp_flags:0x%x\n",
			ioRequest->cdb_status, ioRequest->fcp_status, ioRequest->fcp_rsp_code,
			ioRequest->scsi_comp_flags);
}

static void
qfle3f_show_rport(struct info_str *info, struct qfle3f_rport *target)
{
	struct qfle3fCommand *ioRequest;
    vmk_ListLinks *current, *nextPtr;

	if (!info || !target)
		return;

	copy_info(info, " Target-Port:(%p) cid:%d\n", target, target->fcoe_conn_id);
	copy_info(info,
		"  scsi-target=%16.16lx:%x\n",
		target->portName, target->targetPortID);
	copy_info(info, "      ctx_base:0x%lx conn_id:%d contextID:%d\n",
				target->ctx_base, target->fcoe_conn_id,
				target->contextID);
	copy_info(info, "      flags : ");

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY))
		copy_info(info, "Session_Ready, ");

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFFLOADED))
		copy_info(info, "Offloaded, ");

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_DISABLED))
		copy_info(info, "Disabled, ");

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFLD_REQ_CMPL))
		copy_info(info, "Offload_Req_Complete, ");

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_DESTROYED))
		copy_info(info, "Destroy_Complete, ");

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_CTX_ALLOC_FAILURE))
		copy_info(info, "Ctx_Alloc_Failure, ");

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_UPLD_REQ_COMPL))
		copy_info(info, "Upload_Req_Complete, ");

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_EXPL_LOGO))
		copy_info(info, "Explicit_Logo, ");

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_DEV_LOSS))
		copy_info(info, "Dev loss, ");

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_LOGO_REQ_PENDING))
		copy_info(info, "LOGO Req Pending, ");

	copy_info(info, "\n");

#ifdef IOERROR_DEBUGGING
	copy_info(info, "      ioRetrySessNotReady:%ld ioRetryCmdAllocFailed:%ld ioRetrypostIOReqFailed:%ld\n",
			target->ioRetrySessNotReady, target->ioRetryCmdAllocFailed, target->ioRetrypostIOReqFailed);
#endif

	copy_info(info, "      maxSendQueues:%d maxRequestQueues:%d maxCompletionQueues:%d\n",
			target->maxSendQueues, target->maxRequestQueues, target->maxCompletionQueues);

	copy_info(info, "      SQ_DBELL: prod:0x%x header:0x%x\n",
			target->sq_db.prod, target->sq_db.header.header);

	copy_info(info, "      RX_DBELL: header:0x%x params:0x%x cq_cons:0x%x\n",
			target->rxDb.hdr.header, target->rxDb.params,
			target->rxDb.doorbell_cq_cons);

	copy_info(info, "      SQ:0x%lx prod:0x%x toggle:0x%x mem_sz:0x%x\n",
			GET_DMA_ADDR(target->sqDMA), target->sendQueueProducerIndex,
			target->sendQueueCurrentToggleBit, target->sqMemSize);

	copy_info(info, "      CQ:0x%lx cons:0x%x toggle:0x%x mem_sz:0x%x\n",
			GET_DMA_ADDR(target->cqDMA), target->completionQueueConstIndex,
			target->completionQueueCurrentToggleBit, target->cqMemSize);

	copy_info(info, "      RQ:0x%lx prod:0x%x cons:0x%x mem_sz:0x%x\n",
			GET_DMA_ADDR(target->rqDMA), target->receiveQueueProducerIndex,
			target->receiveQueueConsumerIndex, target->rqMemSize);

	copy_info(info, "      active_ios:%ld flushInProgress:%d\n",
			vmk_AtomicRead64(&target->num_active_ios), target->flushInProgress);

	if (!vmk_ListIsEmpty(&target->activeCommands)) {
		copy_info(info, "      ACTIVE_QUE information\n");
		vmk_SpinlockLock(target->targetLock);
		/* list_for_each_safe(list, tmp, &target->active_cmd_queue) { */
        VMK_LIST_FORALL_SAFE(&target->activeCommands, current, nextPtr) {
			/* cmd = (struct qfle3fCommand *)list; */
            ioRequest = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);
			qfle3f_show_cmd(info, ioRequest);
		}
		vmk_SpinlockUnlock(target->targetLock);
	} else
		copy_info(info, "      ACTIVE_QUE empty\n");

	if (!vmk_ListIsEmpty(&target->elsQueue)) {
		copy_info(info, "      ELS_QUE information\n");
		vmk_SpinlockLock(target->targetLock);
		/* list_for_each_safe(list, tmp, &target->elsQueue) { */
	    VMK_LIST_FORALL_SAFE(&target->elsQueue, current, nextPtr) {
		    ioRequest = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);
			qfle3f_show_cmd(info, ioRequest);
		}
		vmk_SpinlockUnlock(target->targetLock);
	} else
		copy_info(info, "      ELS_QUE empty \n");

	if (!vmk_ListIsEmpty(&target->ioRetireQueue)) {
		copy_info(info, "      IORETIRE_QUE information\n");
		vmk_SpinlockLock(target->targetLock);
		/* list_for_each_safe(list, tmp, &target->ioRetireQueue) { */
	    VMK_LIST_FORALL_SAFE(&target->ioRetireQueue, current, nextPtr) {
			/* ioRequest = (struct qfle3fCommand *)list; */
		    ioRequest = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);
			qfle3f_show_cmd(info, ioRequest);
		}
		vmk_SpinlockUnlock(target->targetLock);
	} else
		copy_info(info, "      IORETIRE_QUE empty\n");

	if (!vmk_ListIsEmpty(&target->activeTaskManagementQueue)) {
		copy_info(info, "      TMF-Q information \n");
		vmk_SpinlockLock(target->targetLock);
	    VMK_LIST_FORALL_SAFE(&target->activeTaskManagementQueue, current, nextPtr) {
		    ioRequest = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);
			qfle3f_show_cmd(info, ioRequest);
		}
		vmk_SpinlockUnlock(target->targetLock);
	} else
		copy_info(info, "      TMF-Q empty\n");

}

#ifdef _QFLE3F_EN_STATS_
static void
qfle3f_show_stats(struct info_str *info, struct qfle3fHBA *hba)
{
	int ret;

	if (!hba->statsBuffer) {
		copy_info(info, "     NO_STATS\n");
		return;
	}

	vmk_SemaLock(&hba->ioctlMutex);
	init_completion(&hba->statRequestDone);
	if (qfle3f_sendStatRequest(hba)) {
		vmk_SemaUnlock(&hba->ioctlMutex);
		copy_info(info, "     NO_STATS\n");
		return;
	}
	ret = wait_for_completion_timeout(&hba->statRequestDone, (2 * vmk_TimerCyclesPerSecond()));
	if (!ret) {
		vmk_SemaUnlock(&hba->ioctlMutex);
		copy_info(info, "     NO_STATS\n");
		return;
	}

	copy_info(info, "     TX_STATS: FCoE(pkt_cnt:%d byte_cnt:%d)"
			"FCP(pkt_cnt:%d)\n",
			hba->statsBuffer->tx_stat.fcoe_tx_pkt_cnt,
			hba->statsBuffer->tx_stat.fcoe_tx_byte_cnt,
			hba->statsBuffer->tx_stat.fcp_tx_pkt_cnt);

	copy_info(info, "     RX_STATS0: FCoE(pkt_cnt:%d byte_cnt:%d)\n",
			hba->statsBuffer->rx_stat0.fcoe_rx_pkt_cnt,
			hba->statsBuffer->rx_stat0.fcoe_rx_byte_cnt);

	copy_info(info, "     RX_STATS1: FCoE(ver_cnt:%d drop_cnt:%d)\n",
			hba->statsBuffer->rx_stat1.fcoe_ver_cnt,
			hba->statsBuffer->rx_stat1.fcoe_rx_drop_pkt_cnt);

	copy_info(info, "     RX_STATS2: fc_crc_cnt:%d eofa_del_cnt:%d "
			"miss_frame_cnt:%d\n",
			hba->statsBuffer->rx_stat2.fc_crc_cnt,
			hba->statsBuffer->rx_stat2.eofa_del_cnt,
			hba->statsBuffer->rx_stat2.miss_frame_cnt);

	copy_info(info, "                seq_timeout:%d drop_seq:%d "
			"fcoe_rx_drop_pkt_cnt:%d fcp_rx_pkt_cnt:%d \n ",
			hba->statsBuffer->rx_stat2.seq_timeout_cnt,
			hba->statsBuffer->rx_stat2.drop_seq_cnt,
			hba->statsBuffer->rx_stat2.fcoe_rx_drop_pkt_cnt,
			hba->statsBuffer->rx_stat2.fcp_rx_pkt_cnt);

	vmk_SemaUnlock(&hba->ioctlMutex);
}
#endif

const char *qfle3f_fip_state_str[] = {
	"FIP_ST_DISABLED", "FIP_ST_LINK_WAIT", "FIP_ST_AUTO",
	"FIP_ST_NON_FIP", "FIP_ST_ENABLED"
};

void copyPortName(struct info_str *info, char *message, vmk_uint8 *name) {
    copy_info(info, "%s = %02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x \n", message,
                name[0], name[1], name[2], name[3],
                name[4], name[5], name[6], name[7]);
}

void copyMACAddress(struct info_str *info, char *message, vmk_uint8 *macAddr) {
    copy_info(info, "%s = %02x:%02x:%02x:%02x:%02x:%02x \n", message,
                macAddr[0], macAddr[1], macAddr[2],
                macAddr[3], macAddr[4], macAddr[5]);
}

extern int qfle3f_enable_cvl_fix;
extern int qfle3f_devloss_tmo;
extern int qfle3f_max_luns;
extern int qfle3f_queue_depth;
extern unsigned long ql_fcoe_debug_level;

static void qfle3f_show_module_params(struct info_str *info)
{
	copy_info(info, " Module Parameters\n");
	copy_info(info, "   qfle3f_debug_level = 0x%lx\n", qfle3f_debug_level);
	copy_info(info, "   ql_fcoe_debug_level = 0x%lx\n", ql_fcoe_debug_level);
	copy_info(info, "   qfle3f_devloss_tmo = %d\n", qfle3f_devloss_tmo);
	copy_info(info, "   qfle3f_max_luns = 0x%x\n", qfle3f_max_luns);
	copy_info(info, "   qfle3f_queue_depth = 0x%x\n", qfle3f_queue_depth);
	copy_info(info, "   qfle3f_enable_cvl_fix = 0x%x\n",
			qfle3f_enable_cvl_fix);
}

static void
qfle3f_show_hba(struct info_str *info, struct qfle3fHBA *hba)
{
	char fw_version[32];

	copy_info(info, "qfle3f-%s:%s v%s over %s\n", "QLogic Offload FCoE",
			hba->sym_name,
			QFLE3F_VERSION, vmk_NameToString(&hba->vmnicName));

	copy_info(info, " %s: v%s\n", QL_FCOE_DRIVER_NAME,
			QL_FCOE_DRIVER_VERSION);

	copy_info(info, " HBA:%p for netdev:%s\n",
				hba,
				vmk_NameToString(&hba->cnic->name));

	copy_info(info, " Host Device Name %s",
				hba->vmhbaName);

	if (hba->type == HBA_PHYSICAL) {
		copy_info(info, " (Physical)\n");

		copy_info(info, " Active NPIV Ports : %d\n",
				hba->vportsInUse);

	} else if (hba->type == HBA_VIRTUAL){
		if (vmk_BitVectorTest(hba->flags2, QFLE3F_NPIV_PREBOOT)) {
			copy_info(info, " (Physical *)\n");
		} else {
			copy_info(info, " (Virtual)\n");
		}
	}

	vmk_NameCopy((vmk_Name *) fw_version,
		&hba->cnic->uplinkSharedData->driverInfo.firmwareVersion);

	copy_info(info, " FW version = %s\n",
			fw_version);
	copy_info(info, "\n");

	if (qfle3f_verbose < QFLE3F_VERBOSE_LPORT)
		return;

	copy_info(info, "     Adapter state: ");
	if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_UP))
		copy_info(info, "UP, ");

	if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_GOING_DOWN))
		copy_info(info, "GOING_DOWN, ");

	if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_LINK_DOWN))
		copy_info(info, "LINK_DOWN, ");

	if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_READY))
		copy_info(info, "READY, ");

	if (vmk_BitVectorTest(hba->hbaState, ADAPTER_STATE_INIT_FAILED))
		copy_info(info, "INIT_FAILED ");

    copy_info(info, "     Vlan (ID=%d) Enabled\n", hba->vlanID);

	copy_info(info, "     %s with cnic\n",
			vmk_BitVectorTest(hba->flags2, QFLE3F_CNIC_REGISTERED)
			? "Registered" : "Not registered");

	copy_info(info, "     initDone: ");

	if (vmk_BitVectorTest(hba->initDone, QFLE3F_FW_INIT_DONE))
		copy_info(info, "FW_INIT_DONE, ");

	if (vmk_BitVectorTest(hba->initDone, QFLE3F_CTLR_INIT_DONE))
		copy_info(info, "CTLR_INIT_DONE, ");

	/* The last element need not have the ,<space> */
	if (vmk_BitVectorTest(hba->initDone, QFLE3F_CREATE_DONE))
		copy_info(info, "CREATE_DONE");

	copy_info(info, "\n");

#ifdef IOERROR_DEBUGGING
	copy_info(info, "      ioRetrySessNotReady:%ld ioRetryCmdAllocFailed:%ld ioRetrypostIOReqFailed:%ld\n",
			hba->ioRetrySessNotReady, hba->ioRetryCmdAllocFailed, hba->ioRetrypostIOReqFailed);
#endif

	copy_info(info, "     LOGO LOGO Race Condition Happened: %ld\n", hba->logoLogoRace);
	copy_info(info, "     IO CMPL - ABORT RACE: %d\n", hba->io_cmpl_abort_race);
	copy_info(info, "     Total number of interrupts: %lu\n", hba->cnic->isr_fcoe_count);


#ifdef _QFLE3F_EN_STATS_
	qfle3f_show_stats(info, hba);

	/* Fix needed. */
	copy_info(info, "    PCI: DID:x%x VID:0x%x SDID:0x%x SVID:0x%x"
			"FUNC:0x%x DEVNO:0x%x\n",
			hba->pci_did, hba->pci_vid, hba->pci_sdid,
			hba->pci_svid, hba->pci_func, hba->pci_devno);
#endif

	copy_info(info, "     num_offload_sess:%d\n", hba->num_ofld_sess);
	copy_info(info, "     cnaInitializeQueue:%ld\n", hba->cnaInitializeQueue);
	copy_info(info, "     cnaCleanupQueue:%ld\n", hba->cnaCleanupQueue);

	/* fip */
	/* qfle3f_show_fip(info, hba); */
	copy_info(info, "\n");

	/* Some Counters*/
	copy_info(info, "  Drop D_ID mismatch: %ld\n",  hba->drop_d_id_mismatch);
	copy_info(info, "  Drop FPMA mismatch: %ld\n",  hba->drop_fpma_mismatch);
	copy_info(info, "  Drop vn_port == 0: %ld\n",  hba->drop_vn_port_zero);
	copy_info(info, "  Drop wrong source MAC %ld\n",  hba->drop_wrong_source_mac);
	copy_info(info, "  Drop ABTS response that has both SEQ/EX CTX set: %ld\n",  hba->drop_abts_seq_ex_ctx_set);
	copy_info(info, "\n");

    copy_info(info, " FIP Information\n");

    copy_info(info, " Source MAC = %02x:%02x:%02x:%02x:%02x:%02x\n",
                hba->dataSourceMacAddress[0], hba->dataSourceMacAddress[1], hba->dataSourceMacAddress[2],
                hba->dataSourceMacAddress[3], hba->dataSourceMacAddress[4], hba->dataSourceMacAddress[5]);

    int i = 0;
    for(i=0; (i < MAX_NUM_FABRICS + MAX_NUM_NPIV_LOGIN); i++) {
        struct ql_fcoe_fabric *Fabric = hba->qlfcoeAdapter->FabricArray[i];
        if(Fabric) {
            copy_info(info, "Fabric[%d] = %p\n", i, Fabric);
            copy_info(info, "Fabric id = %02x:%02x:%02x\n",
                        Fabric->fc_id[0], Fabric->fc_id[1], Fabric->fc_id[2]);
            copyPortName(info, " Switch Port Name", Fabric->SwitchPortWWN);
            copyPortName(info, " FCF Fabric Name", Fabric->FcfFabricName);
            copyPortName(info, " Port WWN", Fabric->PortWWN);

            copyMACAddress(info, " FCF FIP MAC", Fabric->fcf_fip_mac);
            copyMACAddress(info, " FCF FCP MAC", Fabric->fcf_fcp_mac);

            copy_info(info, " E_D_TOV = %ld\n", Fabric->e_d_tov);
            copy_info(info, " R_A_TOV = %ld\n", Fabric->r_a_tov);

			copy_info(info, " LinkKeepaliveInterval = %ld\n", Fabric->LinkKeepaliveInterval);

            copy_info(info, " VLAN ID = %d\n", Fabric->vlan_and_pri & 0xFFF);
        }
    }

	/* lport */
	/* qfle3f_fcinfo_lport(info, lport); */
	copy_info(info, "\n");
}

VMK_ReturnStatus qfle3fGetHbaInfo(vmk_uint64 cookie, void *stringOut)
{
	struct info_str info;
    struct qfle3fHBA *hba = (struct qfle3fHBA *) cookie;
	char *buffer = NULL;

    info.finalLength = MAX_KEY_VAL_STRING_LEN;

	buffer = qfle3f_alloc(info.finalLength+1);
	if (!buffer) {
		vmk_LogMessage("Failed to allocate the buffer for storing Value.");
		return VMK_FAILURE;
	}

	vmk_Memset(buffer, 0, info.finalLength);

	info.buffer      = buffer;
	info.length      = 0;

	if (qfle3f_verbose == QFLE3F_VERBOSE_FC) {
		qfle3f_show_fcinfo(hba, &info);
		goto retval;
	}

	vmk_SpinlockLock(hba->hbaLock);
	qfle3f_show_hba(&info, hba);
	vmk_SpinlockUnlock(hba->hbaLock);

retval:
	copy_info(&info, "\n");
	if (info.length < info.finalLength)
		vmk_StringCopy(stringOut, buffer, info.length);
	else {
		vmk_LogMessage("Too much data! Cannot write full data in mgmtInfo. Data truncated.");
		vmk_StringCopy(stringOut, buffer, info.finalLength);
	}

	qfle3f_free(buffer);
	return VMK_OK;
}

VMK_ReturnStatus qfle3fSetHbaInfo(vmk_uint64 cookie, void *stringIn)
{
    return VMK_OK;
}

VMK_ReturnStatus qfle3fGetTargetInfo(vmk_uint64 cookie, void *stringOut)
{
	struct info_str info;
    struct qfle3fHBA *hba = (struct qfle3fHBA *) cookie;
    struct qfle3f_rport *target;
	char *buffer = NULL;
    int i = 0;

    info.finalLength = MAX_KEY_VAL_STRING_LEN;

	buffer = qfle3f_alloc(info.finalLength+1);
	if (!buffer) {
		vmk_LogMessage("Failed to allocate the buffer for storing Value.");
		return VMK_FAILURE;
	}

	info.buffer      = buffer;
	info.length      = 0;

	vmk_Memset(buffer, 0, info.finalLength);

	/* rport */
	copy_info(&info, " FC Target-Port List\n");
	for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
		target = hba->targetOffloadList[i];
		if (target) {
			qfle3f_show_rport(&info, target);
		}
	}

	copy_info(&info, "\n");

	if (info.length < info.finalLength)
		vmk_StringCopy(stringOut, buffer, info.length);
	else {
		vmk_LogMessage("Too much data! Cannot write full data in mgmtInfo. Data truncated.");
		vmk_StringCopy(stringOut, buffer, info.finalLength);
	}

	qfle3f_free(buffer);
	return VMK_OK;
}

VMK_ReturnStatus qfle3fSetTargetInfo(vmk_uint64 cookie, void *stringIn) {

    return VMK_OK;
}

VMK_ReturnStatus qfle3fSetNpivInfo(vmk_uint64 cookie, void *stringIn) {

    return VMK_OK;
}

void qfle3fKeyValueInitPerHBA(struct qfle3fHBA *hba)
{
	vmk_Name kvName, vmhbaName;
	VMK_ReturnStatus vmk_stat;
	vmk_MgmtProps mgmtProps;

	qfle3f_log(hba, LOG_INIT, "Enter\n");

	vmk_stat = vmk_NameInitialize(&kvName,
			vmk_ScsiGetAdapterName(hba->scsiAdapter));
	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("vmk_MgmtAddKey ADAPTER is failure\n");
		return;
	}
	vmk_NameInitialize(&kvSig.name, vmk_NameToString(&kvName));

	mgmtProps.modId = vmk_ModuleCurrentID;
	mgmtProps.heapId =  vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
	mgmtProps.sig = &kvSig;
	mgmtProps.cleanupFn = NULL;
	mgmtProps.sessionAnnounceFn = NULL;
	mgmtProps.sessionCleanupFn = NULL;
	mgmtProps.handleCookie = (vmk_uint64) hba;
	vmk_stat = vmk_MgmtInit(&mgmtProps,
                                &hba->kvMgmtHandle);
        if (vmk_stat != VMK_OK) {
                vmk_WarningMessage("vmk_MgmtAddKey ADAPTER is failure\n");
                return;
        }
	vmk_NameInitialize(&vmhbaName, "ADAPTER");
	vmk_stat = vmk_MgmtAddKey(hba->kvMgmtHandle,
		VMK_MGMT_KEY_TYPE_STRING,
		&vmhbaName,
		qfle3fGetHbaInfo,
		qfle3fSetHbaInfo);
	VMK_ASSERT(vmk_stat == VMK_OK);

	vmk_NameInitialize(&vmhbaName, "TARGET");
	vmk_stat = vmk_MgmtAddKey(hba->kvMgmtHandle,
		VMK_MGMT_KEY_TYPE_STRING,
		&vmhbaName,
		qfle3fGetTargetInfo,
		qfle3fSetHbaInfo);
	VMK_ASSERT(vmk_stat == VMK_OK);

	qfle3f_log(hba, LOG_INIT, "Exit\n");
}


VMK_ReturnStatus qfle3fGetDriverInfo(vmk_uint64 cookie, void *stringOut)
{
	struct info_str info;
	char *buffer = NULL;

	info.finalLength = MAX_KEY_VAL_STRING_LEN;

	buffer = qfle3f_alloc(info.finalLength+1);
	if (!buffer) {
		vmk_LogMessage("Failed to allocate the buffer for storing Value.");
		return VMK_FAILURE;
	}

	info.buffer      = buffer;
	info.length      = 0;

	vmk_Memset(buffer, 0, info.finalLength);

	qfle3f_show_module_params(&info);

	copy_info(&info, "\n");
	if (info.length < info.finalLength)
		vmk_StringCopy(stringOut, buffer, info.length);
	else {
		vmk_LogMessage("Too much data! Cannot write full data in mgmtInfo. Data truncated.");
		vmk_StringCopy(stringOut, buffer, info.finalLength);
	}

	qfle3f_free(buffer);
	return VMK_OK;
}

typedef struct {
	const char *name;
	unsigned int *value;
}qfle3f_mod_param;

static qfle3f_mod_param mod_name[] = {
	{ "qfle3f_debug_level=", (unsigned int *)&qfle3f_debug_level },
	{ "ql_fcoe_debug_level=", (unsigned int *)&ql_fcoe_debug_level },
	{ "qfle3f_devloss_tmo=", &qfle3f_devloss_tmo },
	{ "qfle3f_enable_r_a_tov=", &qfle3f_enable_r_a_tov },
	{ "qfle3f_r_a_tov=", &qfle3f_r_a_tov },
	{ "qfle3f_default_vlan=", &qfle3f_default_vlan },
	{ "ql_fip_vlan_retries=", &ql_fip_vlan_retries },
	{ NULL, NULL},
};

VMK_ReturnStatus qfle3fSetDriverInfo(vmk_uint64 cookie, void *stringIn)
{
	char *buffer = (char*) stringIn;
	int length = vmk_Strnlen(stringIn, MAX_KEY_VAL_STRING_LEN);
	int i = 0;

	buffer = qfle3f_alloc(length+2);
	vmk_Strncpy(buffer, stringIn, length);

	buffer[length] = '\n';
	buffer[length+1] = '\0';

	/* Module wide operations */

	for (i = 0; mod_name[i].name != NULL; i++) {
		if (!vmk_Strncmp(buffer, mod_name[i].name,
					vmk_Strnlen(mod_name[i].name, 50)) == 0)
			continue;

		if (vmk_Strncmp(buffer + vmk_Strnlen(mod_name[i].name, 50), "0x", 2) == 0) {
			vmk_Sscanf(buffer + vmk_Strnlen(mod_name[i].name, 50) + 2, "%x",
					mod_name[i].value);
			vmk_LogMessage("New value for %s 0x%x\n",
					mod_name[i].name, *mod_name[i].value);
		} else {
			vmk_Sscanf(buffer + vmk_Strnlen(mod_name[i].name, 50), "%d",
					mod_name[i].value);
			vmk_LogMessage("New value for %s %d\n",
					mod_name[i].name, *mod_name[i].value);
		}

		if (!vmk_Strcmp(mod_name[i].name, "qfle3f_debug_level=")) {
			qfle3f_debug_level = *mod_name[i].value;
		} else if (!vmk_Strcmp(mod_name[i].name, "ql_fcoe_debug_level=")) {
			ql_fcoe_debug_level = *mod_name[i].value;
		} else if (!vmk_Strcmp(mod_name[i].name, "qfle3f_devloss_tmo=")) {
			qfle3f_devloss_tmo = *mod_name[i].value;
		} else if (!vmk_Strcmp(mod_name[i].name, "ql_fip_vlan_retries=")) {
			ql_fip_vlan_retries = *mod_name[i].value;
		} else if (!vmk_Strcmp(mod_name[i].name, "qfle3f_default_vlan=")) {
			qfle3f_default_vlan = *mod_name[i].value;
		} else {
			vmk_LogMessage("Invalid parameter to set");
		}
	}

	qfle3f_free(buffer);
    return VMK_OK;

}


VMK_ReturnStatus qfle3fKeyValueInit() {
	vmk_Name kvName, vmhbaName;
	VMK_ReturnStatus vmk_stat;
	vmk_MgmtProps mgmtProps;

	vmk_LogMessage("%s:%d :: Enter", __func__, __LINE__);

	vmk_stat = vmk_NameInitialize(&kvName, "MOD_PARM_FCOE");

	vmk_NameInitialize(&kvSig.name, vmk_NameToString(&kvName));

	mgmtProps.modId = vmk_ModuleCurrentID;
	mgmtProps.heapId =  vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
	mgmtProps.sig = &kvSig;
	mgmtProps.cleanupFn = NULL;
	mgmtProps.sessionAnnounceFn = NULL;
	mgmtProps.sessionCleanupFn = NULL;
	mgmtProps.handleCookie = 0;
	vmk_stat = vmk_MgmtInit(&mgmtProps,
                                &qfle3fDriverInfo.kvMgmtModuleParm);
	VMK_ASSERT(vmk_stat == VMK_OK);

	vmk_NameInitialize(&vmhbaName, "DRIVERINFO");
	vmk_stat = vmk_MgmtAddKey(qfle3fDriverInfo.kvMgmtModuleParm,
		VMK_MGMT_KEY_TYPE_STRING,
		&vmhbaName,
		qfle3fGetDriverInfo,
		qfle3fSetDriverInfo);

	return vmk_stat;
}
