/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

/*
 * qfle3f_target.c: QLogic 10 Gigabit ESX Ethernet FCoE offload driver.
 */

#include "qfle3f.h"
#include "qfle3f_vmk.h"

#include "ql_fcoe_mgmt/ql_fcoe_session.h"
#include "ql_fcoe_mgmt/ql_fcoe_fabric.h"
#include "ql_fcoe_mgmt/FC_CT.h"

extern int qfle3f_devloss_tmo;

void qfle3f_uploadTimer(struct qfle3f_rport *target);
void qfle3f_offloadTimer(struct qfle3f_rport *target);
static int qfle3f_initializeTarget(struct qfle3fHBA *hba,
				struct qfle3f_rport *target);
static int qfle3f_allocSessionResources(struct qfle3fHBA *hba,
			      struct qfle3f_rport *target);
static void qfle3f_freeSessionResources(struct qfle3fHBA *hba,
			      struct qfle3f_rport *target);

void qfle3f_uploadTimer(struct qfle3f_rport *target)
{
	struct qfle3fHBA *hba   = target->hba;
	VMK_ReturnStatus status;

	qfle3f_log(hba, LOG_SESS, "upldTimer - Upload compl not received!!");
	/* fake upload completion */
	vmk_BitVectorClear(target->flags, QFLE3F_FLAG_OFFLOADED);

	vmk_BitVectorSet(target->flags, QFLE3F_FLAG_UPLD_REQ_COMPL);
	qfle3f_log(hba, LOG_SESS, "Waking up uploadWait event");
	status = vmk_WorldWakeup((vmk_WorldEventID) &(target->uploadWait));
	if(status != VMK_OK) {
		qfle3f_err(hba, "Could not wake the world waiting on"
					"event uploadWait: %s.",
					vmk_StatusToString(status));
	}
}

vmk_Bool checkOffloadReqComplFlag(void *data) {
    struct qfle3f_rport *target = (struct qfle3f_rport*) data;

    return vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFLD_REQ_CMPL);
}

void qfle3f_offloadTimer(struct qfle3f_rport *target)
{
	struct qfle3fHBA *hba   = target->hba;
	VMK_ReturnStatus status;

	qfle3f_log(hba, LOG_SESS, "entered qfle3f_offloadTimer");
	/* NOTE: This function should never be called, as
	 * offload should never timeout
	 */
	/*
	 * If the timer has expired, this session is dead
	 * Clear offloaded flag and logout of this device.
	 * Since OFFLOADED flag is cleared, this case
	 * will be coninitiatorPortIDered as offload error and the
	 * port will be logged off, and conn_id, session
	 * resources are freed up in qfle3f_offloadSession
	 */
	vmk_BitVectorClear(target->flags, QFLE3F_FLAG_OFFLOADED);
	vmk_BitVectorSet(target->flags, QFLE3F_FLAG_OFLD_REQ_CMPL);
	status = vmk_WorldWakeup((vmk_WorldEventID)&(target->offloadWait));
	if(status != VMK_OK) {
		qfle3f_err(hba, "Could not wake the world waiting on"
			"event offloadWait: %s.", vmk_StatusToString(status));
	}
}

static void qfle3f_offloadSession(struct qfle3fHBA *hba,
					struct qfle3f_rport *target)
{
	int rval;
	int i = 0;
	VMK_ReturnStatus status;

	/* Initialize qfle3f_rport */
	/* NOTE: target is already bzero'd */
	rval = qfle3f_initializeTarget(hba, target);
	if (rval) {
		qfle3f_err(hba, "Failed to allocate conn id for  targetPortID (%6x)",
			   target->targetPortID);
		goto target_init_err;
	}

	/* Allocate session resources */
	rval = qfle3f_allocSessionResources(hba, target);
	if (rval) {
		qfle3f_err(hba, "Failed to allocate resources");
		goto ofld_err;
	}

	/*
	 * Initialize FCoE session offload process.
	 * Upon completion of offload process add
	 * rport to list of rports
	 */
retry_ofld:
	vmk_BitVectorClear(target->flags, QFLE3F_FLAG_OFLD_REQ_CMPL);
	rval = qfle3f_sendSessionOffloadRequest(hba, target);
	if (rval) {
		qfle3f_err(hba, "ofld_req failed");
		goto ofld_err;
	}

    status = qfle3f_sleepWithCondition(hba, (void *)target,
                                (vmk_WorldEventID)&(target->offloadWait),
                                VMK_LOCK_INVALID,
                                QFLE3F_FW_TIMEOUT,
                                "Waiting for session offload", checkOffloadReqComplFlag);
	if(status != VMK_OK) {
		qfle3f_err(hba, "ERROR! Session Offloaded Request timedout: %s.",
						vmk_StatusToString(status));
		qfle3f_offloadTimer(target);
	}

	if (!(vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFFLOADED))) {
		if (vmk_BitVectorAtomicTestAndClear(target->flags,
						QFLE3F_FLAG_CTX_ALLOC_FAILURE)) {
			qfle3f_log(hba, LOG_SESS, "ctx_alloc_failure, "
				"retry ofld..%d", i++);
			vmk_WorldSleep(1000 * VMK_USEC_PER_MSEC);
			if (i > 3) {
				i = 0;
				goto ofld_err;
			}
			goto retry_ofld;
		}
		goto ofld_err;
	}
	if (qfle3f_mapDoorbell(target)) {
		qfle3f_err(hba, "map doorbell failed - no mem");
		/* upload will take care of cleaning up sess resc */
		goto ofld_err;
	}
	return;

ofld_err:
	/* couldn't offload the session. log off from this rport */
	qfle3f_err(hba, "qfle3f_offloadSession - offload error");
	/* Free session resources before logging off the 'rport' */
	qfle3f_freeSessionResources(hba, target);
target_init_err:
    qfle3f_err(hba, "StartFabricLogout: targetPortID=%0x, WWNN=%lx, WWPN=%lx",
                    target->targetPortID, target->nodeName, target->portName);
    StartFabricLogout(target->Sess->Fabric, 1);
}

void qfle3f_flushActiveIOs(struct qfle3f_rport *target)
{
	struct qfle3fCommand *ioRequest;
	vmk_ListLinks *current, *nextPtr;
	int rc;
	int i = 0;
	struct qfle3fHBA *hba   = target->hba;
	struct cnic_dev *cnic = hba->cnic;
	VMK_ReturnStatus status;

	qfle3f_notice(hba, "Initiating flush_active_ios - 0x%lx, targetPortID = 0x%x",
		vmk_AtomicRead64(&target->num_active_ios), target->targetPortID);

	vmk_SpinlockLock(target->targetLock);
	target->flushInProgress = 1;

	VMK_LIST_FORALL_SAFE(&target->activeCommands, current, nextPtr) {
		i++;
		ioRequest = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);
		vmk_ListRemove(&ioRequest->link);

		ioRequest->on_active_queue = 0;
		qfle3f_log(hba, LOG_IOERR, "cmd_queue cleanup - xid = 0x%x", ioRequest->xid);

		if(ql_vmk_cancel_delayed_work(&ioRequest->timeout_work) == VMK_OK) {
			if (vmk_BitVectorAtomicTestAndClear(ioRequest->req_flags,
						QFLE3F_FLAG_EH_ABORT)) {
				/* Handle eh_abort timeout */
				qfle3f_log(hba, LOG_IOERR, "eh_abort for IO "
						"with oxid = 0x%x "
					"cleaned up", ioRequest->xid);
				vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
			    status = vmk_WorldWakeup((vmk_WorldEventID)&(ioRequest->tm_done));
				if(status != VMK_OK) {
					qfle3f_err(hba, "Could not wake the world waiting on"
						"tm_done: %s.",
						vmk_StatusToString(status));
				}
			}
			ql_vmk_ref_put(&ioRequest->refcount,
				 qfle3f_commandRelease, ioRequest); /* drop timer hold */
		}

		vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_IO_COMPL);
		vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_IO_CLEANUP);
		/* Do not issue cleanup when disable request failed */
		if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_DISABLE_FAILED) ||
				cnic->fw_recov_in_progress)
			qfle3f_processCleanupCompletion(ioRequest, ioRequest->task, 0);
		else {
			rc = qfle3f_initiateCleanup(ioRequest);
			//BUG_ON(rc);
		}
	}

	//list_for_each_safe(list, tmp, &target->activeTaskManagementQueue) {
	VMK_LIST_FORALL_SAFE(&target->activeTaskManagementQueue, current, nextPtr) {
		i++;
		//ioRequest = (struct qfle3fCommand *)list;
		//list_del_init(&ioRequest->link);
		ioRequest = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);
		vmk_ListRemove(&ioRequest->link);

		ioRequest->on_tmf_queue = 0;
		qfle3f_log(hba, LOG_IOERR, "tm_queue cleanup");
		if (ioRequest->wait_for_comp) {
			vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
            status = vmk_WorldWakeup((vmk_WorldEventID)&(ioRequest->tm_done));
			if(status != VMK_OK) {
				qfle3f_err(hba, "Could not wake the world waiting on"
					"tm_done: %s.",
					vmk_StatusToString(status));
			}
		}
	}

	VMK_LIST_FORALL_SAFE(&target->elsQueue, current, nextPtr) {
		i++;
		ioRequest = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);
		vmk_ListRemove(&ioRequest->link);

		ioRequest->on_active_queue = 0;

		qfle3f_log(hba, LOG_IOERR, "elsQueue cleanup - xid = 0x%x", ioRequest->xid);

		if (ql_vmk_cancel_delayed_work(&ioRequest->timeout_work) == VMK_OK) {
			ql_vmk_ref_put(&ioRequest->refcount,
				 qfle3f_commandRelease, ioRequest); /* drop timer hold */

			// Doing the same thing that we do in the commandTimeout Routine.
			if (vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags,
							QFLE3F_FLAG_ELS_DONE))
				continue;

			/* Indicate the cb_func that this ELS is timed out */
			vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_ELS_TIMEOUT);

			if ((ioRequest->cb_func) && (ioRequest->cb_arg)) {
				ioRequest->cb_func(ioRequest->cb_arg);
				ioRequest->cb_arg = NULL;
			}
		}

		rc = qfle3f_initiateCleanup(ioRequest);
		//BUG_ON(rc);
	}

	VMK_LIST_FORALL_SAFE(&target->abtsQueue, current, nextPtr) {
		i++;
		ioRequest = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);
		vmk_ListRemove(&ioRequest->link);

		qfle3f_log(hba, LOG_IOERR, "abtsQueue - xid = 0x%x", ioRequest->xid);

		ql_vmk_ref_put(&ioRequest->refcount,
					qfle3f_commandRelease, ioRequest); /* drop timer hold */
	}

	VMK_LIST_FORALL_SAFE(&target->ioRetireQueue, current, nextPtr) {
		i++;
		//ioRequest = (struct qfle3fCommand *)list;
		//list_del_init(&ioRequest->link);
		ioRequest = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);
		vmk_ListRemove(&ioRequest->link);

		qfle3f_log(hba, LOG_IOERR, "retire_queue flush - xid = 0x%x", ioRequest->xid);

		if (ql_vmk_cancel_delayed_work(&ioRequest->timeout_work) == VMK_OK) {
			if (vmk_BitVectorAtomicTestAndClear(ioRequest->req_flags,
						QFLE3F_FLAG_EH_ABORT)) {
				/* Handle eh_abort timeout */
				qfle3f_log(hba, LOG_IOERR, "eh_abort for IO "
						"in retire_q");
				if (ioRequest->wait_for_comp) {
					vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
                    status = vmk_WorldWakeup((vmk_WorldEventID)&ioRequest->tm_done);
					if(status != VMK_OK) {
						qfle3f_err(hba, "Could not wake the world waiting on"
									"tm_done: %s.",
									vmk_StatusToString(status));
					}
				}
			}
			ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
		}

		vmk_BitVectorClear(ioRequest->req_flags, QFLE3F_FLAG_ISSUE_RRQ);
	}

	qfle3f_log(hba, LOG_IOERR, "IOs flushed = %d", i);
	vmk_SpinlockUnlock(target->targetLock);

	while (vmk_AtomicRead64(&target->num_active_ios) != 0)
		vmk_WorldSleep(100 * VMK_USEC_PER_MSEC);

    qfle3f_notice(hba, "CLEANUP on port 0x%x: active_ios = 0x%lx",
			target->targetPortID, vmk_AtomicRead64(&target->num_active_ios));

	vmk_SpinlockLock(target->targetLock);
	target->flushInProgress = 0;
	vmk_SpinlockUnlock(target->targetLock);
}

vmk_Bool checkUploadReqComplFlag(void *data) {
    struct qfle3f_rport *target = (struct qfle3f_rport*) data;

    return vmk_BitVectorTest(target->flags, QFLE3F_FLAG_UPLD_REQ_COMPL);
}

void qfle3f_uploadSession(struct qfle3fHBA *hba,
					struct qfle3f_rport *target)
{
	VMK_ReturnStatus status;

	qfle3f_log(hba, LOG_SESS, "upload_session: active_ios = 0x%lx",
		vmk_AtomicRead64(&target->num_active_ios));

	/*
	 * Called with hba->hbaMutex held.
	 * This is a blocking call
	 */
	vmk_BitVectorClear(target->flags, QFLE3F_FLAG_UPLD_REQ_COMPL);
	status = qfle3f_sendSessionDisableRequest(hba, target);

	if (status == VMK_OK) {
		qfle3f_log(hba, LOG_SESS, "waiting for disable compl");
		status = qfle3f_sleepWithCondition(hba, (void *)target,
				(vmk_WorldEventID)&(target->uploadWait),
				VMK_LOCK_INVALID,
				QFLE3F_FW_TIMEOUT,
				"Waiting for session upload", checkUploadReqComplFlag);
	}

    if (status != VMK_OK)
        qfle3f_err(hba, "ERROR! Session Destroy Timedout %s",
						vmk_StatusToString(status));

	/*
	 * traverse thru the active_q and tmf_q and cleanup
	 * IOs in these lists
	 */
	qfle3f_log(hba, LOG_SESS, "flush/upload");
	qfle3f_flushActiveIOs(target);

	/* Issue destroy KWQE */
	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_DISABLED)) {
		qfle3f_log(hba, LOG_SESS, "send destroy req");
		vmk_BitVectorClear(target->flags, QFLE3F_FLAG_UPLD_REQ_COMPL);
		qfle3f_sendSessionDestroyRequest(hba, target);

		qfle3f_log(hba, LOG_SESS, "waiting for destroy compl");
        status = qfle3f_sleepWithCondition(hba, (void *)target,
                                (vmk_WorldEventID)&(target->uploadWait),
                                VMK_LOCK_INVALID,
                                QFLE3F_FW_TIMEOUT,
                                "Waiting for session upload", checkUploadReqComplFlag);
		if(status != VMK_OK)
			qfle3f_err(hba, "ERROR! Session Destroy Timedout %s",
						vmk_StatusToString(status));

		if (!(vmk_BitVectorTest(target->flags, QFLE3F_FLAG_DESTROYED)))
			qfle3f_err(hba, "ERROR!! destroy timed out");

		qfle3f_log(hba, LOG_SESS, "destroy wait complete flags.");
	} else {
		qfle3f_err(hba, "ERROR!! DISABLE req timed out, destroy "
			   "not sent to FW");
	}


	/* Free session resources */
	qfle3f_freeSessionResources(hba, target);
}

static int qfle3f_initializeTarget(struct qfle3fHBA *hba,
				struct qfle3f_rport *target)
{
	struct b577xx_doorbell_set_prod *sq_db = &target->sq_db;
	struct b577xx_fcoe_rx_doorbell *rxDb = &target->rxDb;
	vmk_Name name;

	qfle3f_log(hba, LOG_SESS, "entered qfle3f_initializeTarget target = %p", target);

	if (hba->num_ofld_sess >= QFLE3_FCOE_NUM_CONNECTIONS) {
		qfle3f_log(hba, LOG_IOERR, "exceeded max sessions. logoff this target");
		target->fcoe_conn_id = -1;
		return -1;
	}

	/* REVISIT: for configurable WQEs */
	target->maxSendQueues = QFLE3F_SQ_WQES_MAX;
	target->maxRequestQueues = QFLE3F_RQ_WQES_MAX;
	target->maxCompletionQueues = QFLE3F_CQ_WQES_MAX;
	vmk_AtomicWrite64(&target->freeSendQueues, QFLE3F_SQ_WQES_MAX);

	/* Initialize the toggle bit */
	target->sendQueueCurrentToggleBit = 1;
	target->completionQueueCurrentToggleBit = 1;
	target->sendQueueProducerIndex = 0;
	target->completionQueueConstIndex = 0;
	target->receiveQueueProducerIndex = 0x8000;
	target->receiveQueueConsumerIndex = 0;
	vmk_AtomicWrite64(&target->num_active_ios, 0);

	/* initialize sq doorbell */
	sq_db->header.header = B577XX_DOORBELL_HDR_DB_TYPE;
	sq_db->header.header |= B577XX_FCOE_CONNECTION_TYPE <<
					B577XX_DOORBELL_HDR_CONN_TYPE_SHIFT;
	/* initialize rx doorbell */
	rxDb->hdr.header = ((0x1 << B577XX_DOORBELL_HDR_RX_SHIFT) |
			  (0x1 << B577XX_DOORBELL_HDR_DB_TYPE_SHIFT) |
			  (B577XX_FCOE_CONNECTION_TYPE << 
				B577XX_DOORBELL_HDR_CONN_TYPE_SHIFT));
	rxDb->params = (0x2 << B577XX_FCOE_RX_DOORBELL_NEGATIVE_ARM_SHIFT) |
		     (0x3 << B577XX_FCOE_RX_DOORBELL_OPCODE_SHIFT);

	vmk_NameFormat(&name, "target lock %d-%d", hba->instance, target->fcoe_conn_id);
	qfle3f_vmk_spin_lock_init(&target->targetLock, LOCK_RANK_MEDIUM,
								vmk_NameToString(&name));
	vmk_NameFormat(&name, "cq lock %d-%d", hba->instance, target->fcoe_conn_id);
	qfle3f_vmk_spin_lock_init(&target->completionQueueLock, LOCK_RANK_LOWEST,
								vmk_NameToString(&name));

	/* Initialize active_cmd_queue list */
	vmk_ListInit(&target->activeCommands);

	/* Initialize IO retire queue */
	vmk_ListInit(&target->ioRetireQueue);

	vmk_ListInit(&target->elsQueue);

	/* Initialize activeTaskManagementQueue list */
	vmk_ListInit(&target->activeTaskManagementQueue);

	vmk_ListInit(&target->abtsQueue);
	//init_waitqueue_head(&target->offloadWait);
	//init_waitqueue_head(&target->uploadWait);

	return 0;
}

struct qfle3f_rport *qfle3f_remotePortAdd(void *fcoe_drv_handle,
    struct ql_fcoe_session *Sess)
{
	//VMK_ReturnStatus status;
	struct qfle3f_rport *target = NULL;
	struct qfle3fHBA *hba = (struct qfle3fHBA *) fcoe_drv_handle;
	//vmk_uint32 rport_inst = 0;
	//vmk_uint32 hba_inst = 0;
	vmk_uint32 fc_id_temp = 0;
	vmk_uint64 wwnn_temp = 0;
	vmk_uint64 wwpn_temp = 0;
	vmk_Name name;

	fc_id_temp = ((((vmk_uint32)Sess->fc_id[0]) << 16) |
	    (((vmk_uint32)Sess->fc_id[1]) << 8) |
	    (vmk_uint32)Sess->fc_id[2]);

	wwnn_temp = (((vmk_uint64)Sess->NodeWWN[0] << 56) |
	    ((vmk_uint64)Sess->NodeWWN[1] << 48) |
	    ((vmk_uint64)Sess->NodeWWN[2] << 40) |
	    ((vmk_uint64)Sess->NodeWWN[3] << 32) |
	    ((vmk_uint64)Sess->NodeWWN[4] << 24) |
	    ((vmk_uint64)Sess->NodeWWN[5] << 16) |
	    ((vmk_uint64)Sess->NodeWWN[6] << 8) |
	    (vmk_uint64)Sess->NodeWWN[7]);

	wwpn_temp = (((vmk_uint64)Sess->PortWWN[0] << 56) |
	    ((vmk_uint64)Sess->PortWWN[1] << 48) |
	    ((vmk_uint64)Sess->PortWWN[2] << 40) |
	    ((vmk_uint64)Sess->PortWWN[3] << 32) |
	    ((vmk_uint64)Sess->PortWWN[4] << 24) |
	    ((vmk_uint64)Sess->PortWWN[5] << 16) |
	    ((vmk_uint64)Sess->PortWWN[6] << 8) |
	    (vmk_uint64)Sess->PortWWN[7]);

	qfle3f_err(hba, "targetPortID=%0x, WWNN=%lx, WWPN=%lx",
	    fc_id_temp, wwnn_temp, wwpn_temp);

	target = qfle3f_targetLookup(hba, wwnn_temp, wwpn_temp);

    qfle3f_err(hba, "target = %p", target);

	if (target == NULL) {
		/* No consistent binding found - create new remote port entry */
		target = qfle3f_alloc(sizeof(struct qfle3f_rport));

		vmk_ListInitElement(&target->scsiScanList);
		vmk_ListInit(&target->fcluns);
		vmk_ListInit(&target->activeCommands);
		vmk_NameFormat(&name, "fclun list lock %d-%d",
							hba->instance, target->vmk_targetID);
		qfle3f_vmk_spin_lock_init(&target->fclunsListLock, LOCK_RANK_HIGHEST,
									vmk_NameToString(&name));
		// Initializes in qfle3f_initializeTarget()
		//qfle3f_vmk_spin_lock_init(&target->targetLock, LOCK_RANK_HIGHEST);
		vmk_NameFormat(&name, "cmd list lock %d-%d",
							hba->instance, target->vmk_targetID);
		qfle3f_vmk_spin_lock_init(&target->commandListLock, LOCK_RANK_HIGHEST,
									vmk_NameToString(&name));
		ql_vmk_ref_init(&target->refcount);
        if(target)
            qfle3f_err(hba, "target(%p)-%x-0x%x refcount = 0x%lx", target,
                        target->vmk_targetID, target->targetPortID,
                        vmk_AtomicRead64(&target->refcount));
		vmk_AtomicWrite64(&target->state, QFLE3F_RPORT_ST_INIT);
		//qfle3f_log_st(shost, target->id, QFLE3F_RPORT_ST_INIT);
		target->flags =
				vmk_BitVectorAlloc(qfle3fDriverInfo.moduleHeapID, QFLE3F_FLAG_MAX_SIZE);
        vmk_BitVectorZap(target->flags);

		target->maxFrameSize = -1;
		target->supportedClasses = BNX_FC_CLASS_UNSPECIFIED;
		target->nodeName = wwnn_temp;
		target->portName = wwpn_temp;
		target->targetPortID = fc_id_temp;
		if (Sess->SessionType == SESSION_TYPE_TARGET)
			target->portType = FCT_TARGET;
		else if (Sess->SessionType == SESSION_TYPE_INITIATOR)
			target->portType = FCT_INITIATOR;
		target->vmk_channel = 0;
		target->fastIOFailTimeout = -1;

		target->hba = hba;

		target->vmk_targetID = -1;
        target->fcoe_conn_id = -1;

		vmk_SemaLock(&hba->hbaMutex);
		hba->num_rport++;
		vmk_SemaUnlock(&hba->hbaMutex);

	}
	else {

/* Comment:
 	Moving the creation of these two locks from
	qfle3f_initializeTarget() to qfle3f_remotePortAdd()(under the condition of
	target==NULL). This way the lock would only be created once and would be there
	for the lifetime of the qfle3f_rport structure.
*/
		target->targetPortID = fc_id_temp;
		qfle3f_log(hba, LOG_INIT, "Setting the new port id: %0x", target->targetPortID);
		qfle3f_log(hba, LOG_INIT, "Destroying targetLock.\n");
		vmk_SpinlockDestroy(target->targetLock);
		qfle3f_log(hba, LOG_INIT, "Destroying completionQueueLock.\n");
		vmk_SpinlockDestroy(target->completionQueueLock);
		/* Clear all flags */
		vmk_BitVectorZap(target->flags);
		ql_vmk_ref_get(&target->refcount);
	}

	target->Sess = Sess;
    Sess->rport = (void *)target;

	return target;
}

/*
 * Note: A ref_put with this function should not be
 * invoked with any of the following held:
 * 1. hba->hbaMutex
 * 2. target->fclunsListLock
 * 3. hba->scsiScanListLock
 * 4. hba->hbaLock
 */
void qfle3f_freeTarget(void *arg)
{
    struct qfle3f_rport *target = (struct qfle3f_rport *) arg;
    struct qfle3fFCLun *tmp_fclun = NULL;
	struct qfle3fHBA *hba = target->hba;
    vmk_ListLinks *current, *next_ptr;

    if (target != NULL) {
        qfle3f_err(hba, "Deleting target: %x  conn_id = 0x%x targetID = 0x%x",
                        target->targetPortID, target->fcoe_conn_id, target->vmk_targetID);

		if (vmk_ListIsEmpty(&target->fcluns) == VMK_FALSE) {
			vmk_SpinlockLock(target->fclunsListLock);
            VMK_LIST_FORALL_SAFE(&target->fcluns, current, next_ptr) {
                tmp_fclun = VMK_LIST_ENTRY(current, struct qfle3fFCLun, list);
                if(!tmp_fclun)
                    continue;
                vmk_ListRemove(&tmp_fclun->list);
                qfle3f_free(tmp_fclun);
                tmp_fclun = NULL;
            }

            vmk_SpinlockUnlock(target->fclunsListLock);
        }

        vmk_SpinlockLock(hba->scsiScanListLock);
        VMK_LIST_FORALL_SAFE(&hba->scsiScanList, current, next_ptr) {
            if (current == &target->scsiScanList) {
                vmk_ListRemove(&target->scsiScanList);
				break;
            }
        }
        vmk_SpinlockUnlock(hba->scsiScanListLock);

		qfle3f_err(hba, "Destroying fclunsListLock.\n");
        vmk_SpinlockDestroy(target->fclunsListLock);
		qfle3f_err(hba, "Destroying targetLock.\n");
        vmk_SpinlockDestroy(target->targetLock);
		qfle3f_err(hba, "Destroying commandListLock.\n");
        vmk_SpinlockDestroy(target->commandListLock);
		qfle3f_err(hba, "Destroying completionQueueLock.\n");
        vmk_SpinlockDestroy(target->completionQueueLock);
	    vmk_SpinlockLock(hba->hbaLock);
        hba->targetOffloadList[target->vmk_targetID] = NULL;
		vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, target->flags);
	    vmk_SpinlockUnlock(hba->hbaLock);

        //ql_vmk_free(rport->dd_data);
        qfle3f_free(target);
		target = NULL;
		vmk_SemaLock(&hba->hbaMutex);
		hba->num_rport--;
		vmk_SemaUnlock(&hba->hbaMutex);
    }
}

void qfle3f_remotePortDelete(struct qfle3f_rport *target)
{
    struct qfle3fHBA *hba = target->hba;
	qfle3f_notice(hba, "remote delete event, targetPortID=%x", target->targetPortID);

	ql_vmk_ref_put(&target->refcount, qfle3f_freeTarget, target);
	return;
}

const char rportEvNone[10] = "None";
const char rportEvReady[10] = "Ready";
const char rportEvFailed[10] = "Failed";
const char rportEvStop[10] = "Stop";
const char rportEvLogo[10] = "Logout";
const char rportEvUnknown[10] = "Unknown";

const char *qfle3f_rportEventToString(enum ql_fcoe_rport_event event) {
    switch(event) {
        case RPORT_EV_NONE: return rportEvNone;
	    case RPORT_EV_READY: return rportEvReady;
	    case RPORT_EV_FAILED: return rportEvFailed;
	    case RPORT_EV_STOP: return rportEvStop;
	    case RPORT_EV_LOGO: return rportEvLogo;
    }
    return rportEvUnknown;
}

/**
 * This event_callback is called after successful completion of libfc
 * initiated target login. qfle3f can proceed with initiating the session
 * establishment.
 */
void qfle3f_rportEventHandler(void *fcoe_drv_handle,
								struct ql_fcoe_session *Sess,
    							enum ql_fcoe_rport_event event)
{
	struct qfle3fHBA *hba = (struct qfle3fHBA *) fcoe_drv_handle;
    struct ql_fcoe_adapter *Adapter = hba->qlfcoeAdapter;
	struct qfle3f_rport *target;
	vmk_uint8 targetPortIDentifier[2];
	VMK_ReturnStatus status;

	target = (struct qfle3f_rport *)Sess->rport;

	qfle3f_err(hba, "Adapter=%p, hba->instance=%d, event=%s",
	    Adapter, hba->instance, qfle3f_rportEventToString(event));

	if(!target) {
		qfle3f_log(hba, LOG_RPORT, "%p : target is NULL", hba);
		return;
	}

	qfle3f_err(hba, "targetPortID=%0x, WWNN=%lx, WWPN=%lx ref=%ld",
        target->targetPortID, target->nodeName, target->portName, vmk_AtomicRead64(&target->refcount));

	switch (event) {
	case RPORT_EV_READY:
		//if (!rport) {
		//	vmk_LogMessage("rport is NULL: ERROR!");
		//	break;
		//}

		if (target->portType != FCT_TARGET) {
			qfle3f_log(hba, LOG_SESS, "%x - not FCP_TARGET not offloading",
			    target->portType);
			break;
		}
		/*
		 * Offlaod process is protected with hba mutex.
		 * Use the same vmk_SemaLock for upload process too
		 */
		vmk_SemaLock(&hba->hbaMutex);

		/* This can happen when ADISC finds the same target */
		if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFFLOADED)) {
			qfle3f_log(hba, LOG_SESS, "0x%x - already offloaded",
				   target->targetPortID);
			vmk_SemaUnlock(&hba->hbaMutex);
			return;
		}

		struct ql_fcoe_fabric *PhysFabric =
				GetFabricByFlags(Adapter,
					FABRIC_FLAGS_NPIV_SUPPORTED | FABRIC_FLAGS_LOGGED_IN |
					FABRIC_FLAGS_NPIV_OWNER,
					FABRIC_FLAGS_NPIV_SUPPORTED | FABRIC_FLAGS_LOGGED_IN |
					FABRIC_FLAGS_NPIV_OWNER);

		if(PhysFabric) {
			vmk_uint32 fcfTimeout = ((PhysFabric->LinkKeepaliveInterval * 5)/2)/VMK_MSEC_PER_SEC;
			if((fcfTimeout <= QFC_MAX_DEV_LOSS_TMO) && (qfle3f_user_overwrote_devloss_val==0)) {
				qfle3f_warning(hba,
							"Setting devLossTimeout value to %d seconds",
							fcfTimeout);
				hba->devLossTimeout = fcfTimeout;
				ReleaseFabricReference(PhysFabric);
			}
		}

		/*
		 * Offload the session. This is a blocking call, and will
		 * wait until the session is offloaded.
		 */
		qfle3f_offloadSession(hba, target);

		qfle3f_log(hba, LOG_SESS, "OFFLOAD num_ofld_sess = %d",
			hba->num_ofld_sess);

		if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFFLOADED)) {
			/*
			 * Session is offloaded and enabled. Map
			 * doorbell register for this target
			 */
			qfle3f_log(hba, LOG_SESS, "sess offloaded, targetPortID = 0x%x",
				target->targetPortID);
			/* This counter is protected with hba mutex */
			hba->num_ofld_sess++;

			vmk_BitVectorSet(target->flags, QFLE3F_FLAG_SESSION_READY);
			qfle3f_cancelDevlossTimer(target, VMK_FALSE);

			vmk_BitVectorClear(target->flags, QFLE3F_FLAG_DEV_LOSS);
			vmk_BitVectorClear(target->flags, QFLE3F_FLAG_EXPL_LOGO);
			qfle3f_log(hba, LOG_SESS, "Queuing target =0x%x in scsi scan list",
				target->vmk_targetID);
			qfle3f_queue_scsi_scan(target, VMK_TRUE);
			qfle3f_log(hba, LOG_SESS, "Waking scsiScanWorldID = 0x%x", hba->scsiScanWorldID);
			vmk_AtomicWrite64(&hba->wakeupPending, VMK_TRUE);
			status = vmk_WorldForceWakeup(hba->scsiScanWorldID);
			qfle3f_log(hba, LOG_SESS, "scsiScanWorldID status = %s", vmk_StatusToString(status));
		} else {
			/*
			 * Offload or enable would have failed.
			 * In offload/enable completion path, the
			 * rport would have already been removed
			 */
			qfle3f_log(hba, LOG_SESS, "Port is being logged off as "
				   "offloaded flag not set");
		}
		vmk_SemaUnlock(&hba->hbaMutex);
		break;
	case RPORT_EV_LOGO:
	case RPORT_EV_FAILED:
	case RPORT_EV_STOP:
		targetPortIDentifier[0] = ((target->targetPortID & 0xFF0000) >> 16);
		targetPortIDentifier[1] = ((target->targetPortID & 0x00FF00) >> 8);
		targetPortIDentifier[2] = (target->targetPortID & 0x0000FF);
		if (targetPortIDentifier[0] == 0xFF &&
			  targetPortIDentifier[1] == 0xFF &&
			  targetPortIDentifier[2] == FC_CT_GSTYPE_DIRECTORY_SERVER)
			break;

		vmk_SemaLock(&hba->hbaMutex);
		/*
		 * Perform session upload. Note that rdata->peers is already
		 * removed from disc->rports list before we get this event.
		 */

		qfle3f_log(hba, LOG_SESS, "target = %p targetPortID = %x",
			target, target->targetPortID);
		if (!(vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFFLOADED))) {
			qfle3f_log(hba, LOG_SESS, "rport_event_hdlr: target "
				   " was never offloaded");
			vmk_SemaUnlock(&hba->hbaMutex);
			break;
		}
		vmk_BitVectorClear(target->flags, QFLE3F_FLAG_SESSION_READY);

		qfle3f_log(hba, LOG_IOERR,  "rport_event_hdlr:"
			"target = 0x%p, targetPortID = %x",
			target, target->targetPortID);

		qfle3f_uploadSession(hba, target);
		hba->num_ofld_sess--;
		qfle3f_log(hba, LOG_SESS, "UPLOAD num_ofld_sess = %d",
			hba->num_ofld_sess);
		/*
		 * Set the connection id to -1 so we know if we ever assigned
		 * one to the rport.
		 */
		qfle3f_removeConn(hba, target);

		/*
		 * Try to wake up the linkdown wait thread. If num_ofld_sess
		 * is 0, the waiting therad wakes up
		 */
		if ((hba->wait_for_link_down) &&
		    (hba->num_ofld_sess == 0)) {
			status = vmk_WorldWakeup((vmk_WorldEventID) &(hba->shutdownWait));
			if(status != VMK_OK) {
				qfle3f_err(hba, "Could not wake the world waiting on"
					"shutdownWait: %s.",
					vmk_StatusToString(status));
			}
		}

		if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_EXPL_LOGO)) {
			//qfle3f_err(hba, "Relogin to the target");
			//vmk_SemaLock(&lport->disc.disc_mutex);
			//lport->tt.rport_login(rdata);
			//vmk_SemaUnlock(&lport->disc.disc_mutex);
		}


       /* Set up a timer for 2 seconds
        * After 2 seconds, report scsi mid-layer about
        * change in path.
        */
        qfle3f_scheduleDevlossTimer(target);

		vmk_SemaUnlock(&hba->hbaMutex);

		break;

	case RPORT_EV_NONE:
		break;
	}
}

/*
 * qfle3f_targetLookup() - Lookup a qfle3f_rport by targetPortID
 *
 * @port:  fcoe_port struct to lookup the target port on
 * @targetPortID: The remote port ID to look up
 */
struct qfle3f_rport *qfle3f_targetLookup(struct qfle3fHBA *hba,
		vmk_uint64 wwnn, vmk_uint64 wwpn)
{
	struct qfle3f_rport *target;
	int i;

	vmk_SemaLock(&hba->hbaMutex);
	for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
		target = hba->targetOffloadList[i];
		if (target) {
			qfle3f_log(hba, LOG_SESS, "target PortID=%0x, WWNN=%lx, WWPN=%lx",
					target->targetPortID, target->nodeName, target->portName);
		}
		if ((target)
				&& (target->nodeName == wwnn)
				&& (target->portName == wwpn)) {
			vmk_SemaUnlock(&hba->hbaMutex);
			return target;
		}
	}
	vmk_SemaUnlock(&hba->hbaMutex);

	return NULL;
}

/*
 * qfle3f_targetLookupByPortID() - Lookup a qfle3f_rport by targetPortID
 *
 * @port:  fcoe_port struct to lookup the target port on
 * @targetPortID: The remote port ID to look up
 */
struct qfle3f_rport *qfle3f_targetLookupByPortID(struct qfle3fHBA *hba,
		vmk_uint32 portID)
{
	struct qfle3f_rport *target;
	int i;

	for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
		target = hba->targetOffloadList[i];
		if (target) {
			qfle3f_log(hba, LOG_SESS, "target PortID=%0x, WWNN=%lx, WWPN=%lx",
					target->targetPortID, target->nodeName, target->portName);

			if (target->targetPortID == portID) {
				qfle3f_log(hba, LOG_SESS, "Found target=%p portID=0x%x\n",
						target, target->targetPortID);
				return target;
			}
		}
	}

	return NULL;
}

void qfle3f_removeConn(struct qfle3fHBA *hba, struct qfle3f_rport *target)
{
    VMK_ReturnStatus status;

    qfle3f_log(hba, LOG_SESS,
                    "fcoe_conn_id = %d, vmk_targetID = %d",
                    target->fcoe_conn_id, target->vmk_targetID);

    if(target->fcoe_conn_id == -1) {
        qfle3f_err(hba, "Error: RPORT DELETE called with fcoe_conn_id = %d",
                        target->fcoe_conn_id);
        return;
    }

	vmk_SpinlockLock(hba->hbaLock);

    struct qfle3f_rport *rportTemp;

    status = vmk_HashKeyFind(hba->connIdHashTable,
                    (vmk_HashKey)&target->fcoe_conn_id,
                    (vmk_HashValue)&rportTemp);
    if(status != VMK_OK) {
        qfle3f_err(hba, "conn_id = %d not in hash table status = %s",
                    target->fcoe_conn_id, vmk_StatusToString(status));
    }

    if(status == VMK_OK) {
        status = vmk_HashKeyDelete(hba->connIdHashTable,
                            (vmk_HashKey)&target->fcoe_conn_id,
                            NULL);
        if(status != VMK_OK)
            qfle3f_err(hba, "conn_id = %d vmk_HashKeyDelete failed = %s",
                        target->fcoe_conn_id, vmk_StatusToString(status));
    }

	vmk_SpinlockUnlock(hba->hbaLock);

    target->fcoe_conn_id = -1;
}

/**
 * qfle3f_addConn - add FCOE Connection to hash table
 *
 * @hba:	pointer to adapter structure
 * @target:	pointer to qfle3f_rport structure
 */
void qfle3f_addConn(struct qfle3fHBA *hba,
				struct qfle3f_rport *target)
{
    VMK_ReturnStatus status;
	int i = 0;

    qfle3f_log(hba, LOG_SESS,
                    "fcoe_conn_id = %d, vmk_targetID = %d",
                    target->fcoe_conn_id, target->vmk_targetID);

	vmk_SpinlockLock(hba->hbaLock);

    if(target->vmk_targetID == -1) {
		if(hba->type == HBA_PHYSICAL) {
			for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
				if (hba->targetOffloadList[i] == NULL) {
					target->vmk_targetID = i;
					break;
				}
			}
		} else if(hba->type == HBA_VIRTUAL) {
			struct qfle3fHBA *phba = hba->phba;
			vmk_SpinlockUnlock(hba->hbaLock);
			struct qfle3f_rport *ptarget = qfle3f_targetLookup(phba,
									target->nodeName, target->portName);
			vmk_SpinlockLock(hba->hbaLock);
			if(ptarget) {
				target->vmk_targetID = ptarget->vmk_targetID;
			} else {
				qfle3f_err(hba, "Error: Target is not offloaded for Physical Port.");
				// Find a free slot in physical hba targetOffloadList and use that
				// as the vmk_targetID
				for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
					if (phba->targetOffloadList[i] == NULL) {
						target->vmk_targetID = i;
						break;
					}
				}
			}
		}
		if (target->vmk_targetID == -1) {
			qfle3f_err(hba, "Error: No free slot in targetOffloadList!");
		}
	}
	hba->targetOffloadList[target->vmk_targetID] = target;

    vmk_HashValue val;

    status = vmk_HashKeyFind(hba->connIdHashTable,
                        (vmk_HashKey)&target->fcoe_conn_id,
                        &val);
    if(status == VMK_OK) {
        qfle3f_err(hba, "ERROR: conn_id = %d already in hash table = %s, %p",
                    target->fcoe_conn_id, vmk_StatusToString(status), val);
    }

    if(status != VMK_OK) {
        status = vmk_HashKeyInsert(hba->connIdHashTable,
                            (vmk_HashKey)&target->fcoe_conn_id,
                            (vmk_HashValue)target);
        qfle3f_err(hba, "vmk_HashKeyInsert ret status = %s conn_id = %d",
                    vmk_StatusToString(status), target->fcoe_conn_id);
    }

	vmk_SpinlockUnlock(hba->hbaLock);

    qfle3f_log(hba, LOG_SESS, "fcoe_conn_id = %d, vmk_targetID = %d",
                    target->fcoe_conn_id, target->vmk_targetID);

	return;
}

/**
 *qfle3f_allocSessionResources - Allocate qp resources for the session
 *
 */
static int qfle3f_allocSessionResources(struct qfle3fHBA *hba,
					struct qfle3f_rport *target)
{
	vmk_IOA page;
	int num_pages;
	vmk_uint32 *pbl;

	/* Allocate and map SQ */
	target->sqMemSize = (target->maxSendQueues + 8) * QFLE3F_SQ_WQE_SIZE;
	target->sqMemSize = (target->sqMemSize + (VMK_PAGE_SIZE - 1)) & ~VMK_PAGE_MASK;

	target->sq = qfle3f_dma_alloc(hba,
							&target->sqDMA,
							target->sqMemSize);
	if (!target->sq) {
		vmk_LogMessage("unable to allocate SQ memory %d",
			target->sqMemSize);
		goto mem_alloc_failure;
	}
	vmk_Memset(target->sq, 0, target->sqMemSize);

	/* Allocate and map CQ */
	target->cqMemSize = (target->maxCompletionQueues + 8) * QFLE3F_CQ_WQE_SIZE;
	target->cqMemSize = (target->cqMemSize + (VMK_PAGE_SIZE - 1)) & ~VMK_PAGE_MASK;

	target->cq = qfle3f_dma_alloc(hba,
						&target->cqDMA,
						target->cqMemSize);
	if (!target->cq) {
		vmk_LogMessage("unable to allocate CQ memory %d",
			target->cqMemSize);
		goto mem_alloc_failure;
	}
	vmk_Memset(target->cq, 0, target->cqMemSize);

	/* Allocate and map RQ and RQ PBL */
	target->rqMemSize = target->maxRequestQueues * QFLE3F_RQ_WQE_SIZE;
	target->rqMemSize = (target->rqMemSize + (VMK_PAGE_SIZE - 1)) & ~VMK_PAGE_MASK;

	target->rq = qfle3f_dma_alloc(hba,
					&target->rqDMA, target->rqMemSize);
	if (!target->rq) {
		vmk_LogMessage("unable to allocate RQ memory %d",
			target->rqMemSize);
		goto mem_alloc_failure;
	}
	vmk_Memset(target->rq, 0, target->rqMemSize);

	target->rqPblSize = (target->rqMemSize / VMK_PAGE_SIZE) * sizeof(void *);
	target->rqPblSize = (target->rqPblSize + (VMK_PAGE_SIZE - 1)) & ~VMK_PAGE_MASK;

	target->rqPbl = qfle3f_dma_alloc(hba,
						&target->rqPblDMA, target->rqPblSize);
	if (!target->rqPbl) {
		vmk_LogMessage("unable to allocate RQ PBL %d",
			target->rqPblSize);
		goto mem_alloc_failure;
	}

	vmk_Memset(target->rqPbl, 0, target->rqPblSize);
	num_pages = target->rqMemSize / VMK_PAGE_SIZE;
	page = GET_DMA_ADDR(target->rqDMA);
	pbl = (vmk_uint32 *)target->rqPbl;

	while (num_pages--) {
		*pbl = U64_LO(page);
		pbl++;
		*pbl = U64_HI(page);
		pbl++;
		page += VMK_PAGE_SIZE;
	}

	/* Allocate and map XFERQ */
	target->xferqMemSize = (target->maxSendQueues + 8) * QFLE3F_XFERQ_WQE_SIZE;
	target->xferqMemSize = (target->xferqMemSize + (VMK_PAGE_SIZE -1)) &
			       ~VMK_PAGE_MASK;

	target->xferq = qfle3f_dma_alloc(hba,
						&target->xferqDMA,
						target->xferqMemSize);
	if (!target->xferq) {
		vmk_LogMessage("unable to allocate XFERQ %d",
			target->xferqMemSize);
		goto mem_alloc_failure;
	}
	vmk_Memset(target->xferq, 0, target->xferqMemSize);

	/* Allocate and map CONFQ & CONFQ PBL */
	target->confqMemSize = (target->maxSendQueues + 8) * QFLE3F_CONFQ_WQE_SIZE;
	target->confqMemSize = (target->confqMemSize + (VMK_PAGE_SIZE -1)) & 
			       ~VMK_PAGE_MASK;

	target->confq = qfle3f_dma_alloc(hba,
						&target->confqDMA, target->confqMemSize);
	if (!target->confq) {
		vmk_LogMessage("unable to allocate CONFQ %d",
			target->confqMemSize);
		goto mem_alloc_failure;
	}
	vmk_Memset(target->confq, 0, target->confqMemSize);

	target->confqPblSize =
		(target->confqMemSize / VMK_PAGE_SIZE) * sizeof(void *);
	target->confqPblSize =
		(target->confqPblSize + (VMK_PAGE_SIZE -1)) & ~VMK_PAGE_MASK;

	target->confqPbl = qfle3f_dma_alloc(hba,
							&target->confqPblDMA,
							target->confqPblSize);
	if (!target->confqPbl) {
		vmk_LogMessage("unable to allocate CONFQ PBL %d",
			target->confqPblSize);
		goto mem_alloc_failure;
	}

	vmk_Memset(target->confqPbl, 0, target->confqPblSize);
	num_pages = target->confqMemSize / VMK_PAGE_SIZE;
	page = GET_DMA_ADDR(target->confqDMA);
	pbl = (vmk_uint32 *)target->confqPbl;

	while (num_pages--) {
		*pbl = U64_LO(page);
		pbl++;
		*pbl = U64_HI(page);
		pbl++;
		page += VMK_PAGE_SIZE;
	}

	/* Allocate and map ConnDB */
	target->connDbMemSize = sizeof(struct fcoe_conn_db);

	target->connDb = qfle3f_dma_alloc(hba,
						&target->connDbDMA,
						target->connDbMemSize);
	if (!target->connDb) {
		vmk_LogMessage("unable to allocate connDb %d",
						target->connDbMemSize);
		goto mem_alloc_failure;
	}
	vmk_Memset(target->connDb, 0, target->connDbMemSize);

	/* Allocate and map LCQ */
	target->lcqMemSize = (target->maxSendQueues + 8) * QFLE3F_LCQ_WQE_SIZE;
	target->lcqMemSize = (target->lcqMemSize + (VMK_PAGE_SIZE - 1)) &
			     ~VMK_PAGE_MASK;

	target->lcq = qfle3f_dma_alloc(hba,
					&target->lcqDMA,
					target->lcqMemSize);

	if (!target->lcq) {
		vmk_LogMessage("unable to allocate lcq %d",
		       target->lcqMemSize);
		goto mem_alloc_failure;
	}
	vmk_Memset(target->lcq, 0, target->lcqMemSize);

	/* Arm CQ */
	target->connDb->cq_arm.lo = -1;
	target->connDb->rq_prod = 0x8000;

	return 0;

mem_alloc_failure:
	return -1;
}

/**
 * bnx2i_free_session_resc - free qp resources for the session
 *
 * @hba:	adapter structure pointer
 * @target:	qfle3f_rport structure pointer
 *
 * Free QP resources - SQ/RQ/CQ/XFERQ memory and PBL
 */
static void qfle3f_freeSessionResources(struct qfle3fHBA *hba,
						struct qfle3f_rport *target)
{
	qfle3f_log(hba, LOG_SESS, "Freeing up session resources - 0x%x",
		target->targetPortID);

	vmk_SpinlockLock(target->completionQueueLock);
	target->ctx_base = 0;

	/* Free LCQ */
	if (target->lcq) {
		qfle3f_dma_free(hba, &target->lcqDMA);
		target->lcq = NULL;
	}
	/* Free connDB */
	if (target->connDb) {
		qfle3f_log(hba, LOG_SESS, "Freeing connDb");
		qfle3f_dma_free(hba, &target->connDbDMA);
		target->connDb = NULL;
	}
	/* Free confq  and confq pbl */
	if (target->confqPbl) {
		qfle3f_dma_free(hba, &target->confqPblDMA);
		target->confqPbl = NULL;
	}
	if (target->confq) {
		qfle3f_dma_free(hba, &target->confqDMA);
		target->confq = NULL;
	}
	/* Free XFERQ */
	if (target->xferq) {
		qfle3f_dma_free(hba, &target->xferqDMA);
		target->xferq = NULL;
	}
	/* Free RQ PBL and RQ */
	if (target->rqPbl) {
		qfle3f_dma_free(hba, &target->rqPblDMA);
		target->rqPbl = NULL;
	}
	if (target->rq) {
		qfle3f_dma_free(hba, &target->rqDMA);
		target->rq = NULL;
	}
	/* Free CQ */
	if (target->cq) {
		qfle3f_dma_free(hba, &target->cqDMA);
		target->cq = NULL;
	}
	/* Free SQ */
	if (target->sq) {
		qfle3f_dma_free(hba, &target->sqDMA);
		target->sq = NULL;
	}
	vmk_SpinlockUnlock(target->completionQueueLock);
}
