/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#include <qfle3f_vmk.h>
#include <qfle3f.h>

extern struct qfle3fDriverInfo_t qfle3fDriverInfo;

void *qfle3f_alloc(vmk_uint32 size)
{
	void *data;
	int retry_count = 10;

 qfle3f_alloc_retry:
	if (size > VMK_PAGE_SIZE)
		data = vmk_HeapAlign(qfle3fDriverInfo.moduleHeapID,
			size, VMK_PAGE_SIZE);
	else
		data = vmk_HeapAlloc(qfle3fDriverInfo.moduleHeapID, size);

	if (data != NULL)
		vmk_Memset(data, 0, size);
    else {
        vmk_WarningMessage("%s:%d: Allocation Failed.", __func__, __LINE__);
		if (retry_count--) {
			vmk_WarningMessage("%s:%d: Remaining qfle3f_alloc_retry = %d\n",
				__func__, __LINE__, retry_count);
			vmk_WorldSleep(10 * VMK_USEC_PER_MSEC);
			goto qfle3f_alloc_retry;
		}
        vmk_WarningMessage("%s:%d: All retry exhust.", __func__, __LINE__);
	}


	return data;
}

inline void qfle3f_free(void *buffer)
{
	if (buffer)
		vmk_HeapFree(qfle3fDriverInfo.moduleHeapID, buffer);
}

int
qfle3f_dma_map(struct qfle3fHBA *hba, void *mem, struct qla_dmamem *buffer, vmk_uint32 size)
{
   	VMK_ReturnStatus status = VMK_OK;
 	vmk_DMAMapErrorInfo err;

	if (mem == NULL) {
		goto sgCreate;
	}

	vmk_Memset(buffer, 0, sizeof(struct qla_dmamem));

	status = vmk_SgCreateOpsHandle(qfle3fDriverInfo.moduleHeapID,
		&buffer->SgOps_handle, NULL, NULL);

	if (status != VMK_OK)
		goto sgCreate;

	/* Allocate and initialize a Input sg array with machine-addresses */
	status = vmk_SgAllocWithInit(buffer->SgOps_handle,
		&(buffer->MachSg), mem, size);

	if (status != VMK_OK)
		goto sgInit;

	buffer->dma_size = size;
	buffer->virtaddr = mem;

	/* Map the sg array with the DMA Engine of the device */
	status = vmk_DMAMapSg(hba->ioDMAEngine,
		VMK_DMA_DIRECTION_BIDIRECTIONAL, buffer->SgOps_handle,
		buffer->MachSg, &(buffer->MachSg), &err);

	if (status != VMK_OK) {
		qfle3f_warning(hba, "failed to map %p size %d to IO address, %s",
			mem, size, vmk_DMAMapErrorReasonToString(err.reason));
		goto sgAlloc;
	}

#if 0
//SK:
	qfle3f_err(hba, "%s: allocated dma buffer %p, "
		"size = %x, elements = %x, MA0 = %lx, size0 = %x",
		__func__, mem, size, buffer->MachSg->numElems,
		buffer->MachSg->elem[0].ioAddr, buffer->MachSg->elem[0].length);
#endif

	return VMK_OK;

sgAlloc:
	vmk_SgFree(buffer->SgOps_handle, buffer->MachSg);
sgInit:
	vmk_SgDestroyOpsHandle(buffer->SgOps_handle);
sgCreate:
	qfle3f_warning(hba, "Unable to map DMA buffer!");

	return VMK_FAILURE;
}

void
qfle3f_dma_unmap(struct qfle3fHBA *hba, struct qla_dmamem *buffer)
{
	if (buffer->virtaddr == NULL)
		return;
	vmk_DMAUnmapSg(hba->ioDMAEngine, VMK_DMA_DIRECTION_BIDIRECTIONAL,
	    buffer->SgOps_handle, buffer->MachSg);
	vmk_SgFree(buffer->SgOps_handle, buffer->MachSg);
	vmk_SgDestroyOpsHandle(buffer->SgOps_handle);
}

void *
qfle3f_dma_alloc(struct qfle3fHBA *hba, struct qla_dmamem *buffer, vmk_uint32 size)
{
	void *map = NULL;
	int retry_count = 10;

qfle3f_dma_alloc_retry:
	if (size > VMK_PAGE_SIZE)
		map = vmk_HeapAlign(qfle3fDriverInfo.moduleDMAHeapID,
			                size, VMK_PAGE_SIZE);
	else
		map = vmk_HeapAlloc(qfle3fDriverInfo.moduleDMAHeapID, size);

	if (map == NULL) {
        vmk_WarningMessage("%s:%d: Allocation Failed.", __func__, __LINE__);
		if (retry_count--) {
			vmk_WarningMessage("%s:%d: Remaining qfle3f_dma_alloc_retry = %d\n",
				__func__, __LINE__, retry_count);
			vmk_WorldSleep(10 * VMK_USEC_PER_MSEC);
			goto qfle3f_dma_alloc_retry;
		}
        vmk_WarningMessage("%s:%d: All retry exhust.", __func__, __LINE__);
		return map;
    }

	vmk_Memset(map, 0, size);

	retry_count = 10;
qfle3f_dma_alloc_retry1:
	if (qfle3f_dma_map(hba, map, buffer, size) != 0) {
		if (retry_count--) {
			vmk_WarningMessage("%s:%d: Remaining qfle3f_dma_alloc_retry = %d\n",
				__func__, __LINE__, retry_count);
			vmk_WorldSleep(10 * VMK_USEC_PER_MSEC);
			goto qfle3f_dma_alloc_retry1;
		}
        vmk_WarningMessage("%s:%d: All retry exhust.", __func__, __LINE__);

		vmk_HeapFree(qfle3fDriverInfo.moduleDMAHeapID, map);
        vmk_WarningMessage("%s:%d: DMA Map Failed.", __func__, __LINE__);
		map = NULL;
	}

#if 0
	qfle3f_err(hba, "Allocating dma buffer %p, "
		"size = %x, elements = %x, MA0 = %lx, size0 = %x",
		buffer->virtaddr, buffer->dma_size, buffer->MachSg->numElems,
		buffer->MachSg->elem[0].ioAddr, buffer->MachSg->elem[0].length);
#endif

	return map;
}

void
qfle3f_dma_free(struct qfle3fHBA *hba, struct qla_dmamem *buffer)
{
#if 0
	qfle3f_err(hba, "Freeing dma buffer %p, "
		"size = %x, elements = %x, MA0 = %lx, size0 = %x",
		buffer->virtaddr, buffer->dma_size, buffer->MachSg->numElems,
		buffer->MachSg->elem[0].ioAddr, buffer->MachSg->elem[0].length);
#endif
	if (buffer->virtaddr == NULL)
		return;
	qfle3f_dma_unmap(hba, buffer);
	vmk_HeapFree(qfle3fDriverInfo.moduleDMAHeapID, buffer->virtaddr);
	buffer->virtaddr = NULL;
}

void *
qfle3f_dma_slab_alloc(struct qfle3fHBA *hba, vmk_SlabID slab_id,
                struct qla_dmamem *buffer, vmk_uint32 size)
{
	void *map = NULL;
	int retry_count = 10;

qfle3f_dma_slab_alloc_retry:
	map = vmk_SlabAlloc(slab_id);
	if (map == NULL) {
		vmk_WarningMessage("%s:%d: Allocation Failed.", __func__, __LINE__);
		if (retry_count--) {
			vmk_WarningMessage("%s:%d: Remaining qfle3f_dma_slab_alloc_retry = %d\n",
				__func__, __LINE__, retry_count);
			vmk_WorldSleep(10 * VMK_USEC_PER_MSEC);
			goto qfle3f_dma_slab_alloc_retry;
		}
        vmk_WarningMessage("%s:%d: All retry exhust.", __func__, __LINE__);

		return NULL;
	}

qfle3f_dma_slab_alloc_retry1:
	if (qfle3f_dma_map(hba, map, buffer, size) != 0) {
		vmk_WarningMessage("%s:%d: Allocation Failed.", __func__, __LINE__);
		if (retry_count--) {
			vmk_WarningMessage("%s:%d: Remaining qfle3f_dma_slab_alloc = %d\n",
				__func__, __LINE__, retry_count);
			vmk_WorldSleep(10 * VMK_USEC_PER_MSEC);
			goto qfle3f_dma_slab_alloc_retry1;
		}
        vmk_WarningMessage("%s:%d: All retry exhust.", __func__, __LINE__);

		vmk_SlabFree(slab_id, map);
		map = NULL;
	}

	return map;
}

void
qfle3f_dma_slab_free(struct qfle3fHBA *hba, vmk_SlabID slab_id, struct qla_dmamem *buffer)
{
	qfle3f_dma_unmap(hba, buffer);
	vmk_SlabFree(slab_id, buffer->virtaddr);
	buffer->virtaddr = NULL;
}

VMK_ReturnStatus qfle3f_vmk_spin_lock_init(vmk_Lock *lock,
	ql_vmk_lock_rank lock_rank, const char *name)
{
	vmk_SpinlockCreateProps spinlockProps;
	VMK_ReturnStatus vmkStat;
	vmk_Name spinLockName;
	int retry_count = 10;

	vmk_Memset(&spinlockProps, 0, sizeof(spinlockProps));
	spinlockProps.moduleID = vmk_ModuleCurrentID;
	spinlockProps.heapID = vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
	spinlockProps.type = VMK_SPINLOCK;
	spinlockProps.domain = qfle3fDriverInfo.lockDomain;
	spinlockProps.rank = lock_rank;

	vmkStat = vmk_NameInitialize(&spinLockName, (const char *)&name);
	if (vmkStat != VMK_OK) {
		vmk_WarningMessage("%s failed : %s",
				__func__, vmk_StatusToString(vmkStat));
		goto spin_lock_fail;
	}
	spinlockProps.name = spinLockName;

qfle3f_vmk_spin_lock_init_retry:
	vmkStat = vmk_SpinlockCreate(&spinlockProps, lock);

	if (vmkStat != VMK_OK) {
		vmk_WarningMessage("%s failed to create lock: %s",
				__func__, vmk_StatusToString(vmkStat));
		vmk_WarningMessage("%s:%d: vmk_SpinlockCreate Failed.",
				__func__, __LINE__);
		if (retry_count--) {
			vmk_WarningMessage("%s:%d: "
					"Remaining qfle3f_vmk_spin_lock_init_retry = %d\n",
					__func__, __LINE__, retry_count);
			vmk_WorldSleep(10 * VMK_USEC_PER_MSEC);
			goto qfle3f_vmk_spin_lock_init_retry;
		}
		vmk_WarningMessage("%s:%d: All retry exhausted.", __func__, __LINE__);

		goto spin_lock_fail;
	}

spin_lock_fail:
	return vmkStat;
}


void ql_vmk_ref_init(vmk_atomic64 *refcount)
{
    vmk_AtomicWrite64(refcount, 1);
    vmk_AtomicEpilogue();
}

vmk_uint16 ql_vmk_ref_get(vmk_atomic64 *refcount)
{
    if (!vmk_AtomicRead64(refcount)) {
        vmk_WarningMessage("%s: refcount has already reached 0.", __func__);
        return 0;
    }

    vmk_AtomicInc64(refcount);
    vmk_AtomicEpilogue();
    //vmk_WarningMessage("%s:%d :: refcount = 0x%lx",
    //                    __func__, __LINE__, vmk_AtomicRead64(refcount));
    return 1;
}

vmk_uint16 ql_vmk_ref_put(vmk_atomic64 *refcount,
    void (*release)(void *arg), void *arg)
{
    //vmk_WarningMessage("%s:%d :: refcount = 0x%lx",
    //                    __func__, __LINE__, vmk_AtomicRead64(refcount));
    if (release == NULL)
        vmk_AlertMessage("%s: Reference release function cannot be NULL.",
            __func__);
    if (vmk_AtomicRead64(refcount) == 0)
        vmk_AlertMessage("%s: Reference already zero!\n", __func__);

    if (vmk_AtomicReadDec64(refcount) == 1) {
        release(arg);
        return 1;
    }

    return 0;
}

/* Workqueue functions */
VMK_ReturnStatus
ql_vmk_do_singlethread_work(void *data)
{
	ql_vmk_singlethread_workqueue_t *wq = (ql_vmk_singlethread_workqueue_t *)data;

	while (vmk_WorldWait(VMK_EVENT_NONE, VMK_LOCK_INVALID,
			VMK_TIMEOUT_UNLIMITED_MS, "wq thread sleeping") == VMK_OK) {
		ql_vmk_singlethread_workitem_t *item;

		vmk_SpinlockLock(wq->lock);
		/* there could be spurious wakeups */
		while (!vmk_ListIsEmpty(&wq->work_items)) {
			item = VMK_LIST_ENTRY(vmk_ListFirst(&wq->work_items),
				ql_vmk_singlethread_workitem_t, links);
			vmk_ListRemove(&item->links);
			vmk_ListInitElement(&item->links);
			item->workitem_state = INACTIVE;
			vmk_SpinlockUnlock(wq->lock);
			item->handler(item);
			vmk_SpinlockLock(wq->lock);
		}
		vmk_SpinlockUnlock(wq->lock);
	}
	return VMK_OK;
}

void
ql_vmk_init_singlethread_workitem(ql_vmk_singlethread_workitem_t *item,
	ql_vmk_singlethread_workhandler_t handler, void *data1, void *data2)
{
	vmk_ListInitElement(&item->links);
	item->handler = handler;
	item->data1 = data1;
	item->data2 = data2;
	item->workitem_state = INACTIVE;
}

void ql_vmk_singlethread_queue_work(ql_vmk_singlethread_workqueue_t *wq,
	ql_vmk_singlethread_workitem_t *item)
{
	if (!wq || !item)
		return;

	vmk_SpinlockLock(wq->lock);
	if (item->workitem_state == INACTIVE ||
	    item->workitem_state == DELAY_ACTIVE) {
		if (vmk_ListIsUnlinkedElement(&item->links))
			vmk_ListInsert(&item->links, vmk_ListAtRear(&wq->work_items));
		else
			vmk_WarningMessage("Work item was on list but not active!!\n");
		item->workitem_state = ACTIVE;
	}
	vmk_SpinlockUnlock(wq->lock);

	if (vmk_WorldForceWakeup(wq->world_id) != VMK_OK)
		vmk_WarningMessage("Could not wake up %s", wq->name);
}

ql_vmk_singlethread_workqueue_t *
ql_vmk_create_singlethread_workqueue(const char *name, vmk_WorldSchedClass prio)
{
	ql_vmk_singlethread_workqueue_t *wq = NULL;
	VMK_ReturnStatus status;
	vmk_WorldProps wq_props;

	wq = qfle3f_alloc(sizeof(*wq));
	if (wq == NULL) {
		status = VMK_NO_MEMORY;
		vmk_WarningMessage("Unable to allocate WQ");
		goto wq_failed;
	}

	vmk_ListInit(&wq->work_items);

	vmk_StringCopy(wq->name, name, sizeof(wq->name));

	status = qfle3f_vmk_spin_lock_init(&wq->lock, LOCK_RANK_HIGHEST,
		(char *) name);

	if (status != VMK_OK) {
		vmk_WarningMessage("Unable to create sp lock for WQ thread!");
		goto wq_failed;
	}

	vmk_Memset(&wq_props, 0, sizeof(vmk_WorldProps));
#if (VMWARE_ESX_DDK_VERSION > 60000)
		wq_props.heapID = qfle3fDriverInfo.moduleHeapID;
#endif
	wq_props.moduleID = vmk_ModuleCurrentID;
	wq_props.startFunction = (void *)ql_vmk_do_singlethread_work;
	wq_props.data = (void *)wq;
	wq_props.schedClass = prio;
	wq_props.name = wq->name;
	status = vmk_WorldCreate(&wq_props, &wq->world_id);
	if (status != VMK_OK) {
		vmk_WarningMessage("Unable to start WQ thread!");
		goto spinlock_failed;
	}

spinlock_failed:
	if (status != VMK_OK) {
		vmk_SpinlockDestroy(wq->lock);
	}

wq_failed:
	if (status != VMK_OK) {
		qfle3f_free(wq);
		wq = NULL;
	}

	return wq;
}

void
ql_vmk_destroy_singlethread_workqueue(ql_vmk_singlethread_workqueue_t *wq)
{
	VMK_ReturnStatus status;
	status = vmk_WorldDestroy(wq->world_id);
	if(status != VMK_OK) {
		vmk_WarningMessage("word destroy failed for world id : %d. status = %s",
					wq->world_id, vmk_StatusToString(status));
	}
	status = vmk_WorldWaitForDeath(wq->world_id);
	if(status == VMK_OK) {
		vmk_WarningMessage("world %d is definately dead",
					wq->world_id);
	} else {
		vmk_WarningMessage("wait for death of world %d failed. status = %s",
					wq->world_id, vmk_StatusToString(status));
	}
	vmk_SpinlockDestroy(wq->lock);

	qfle3f_free(wq);
}

void ql_vmk_init_singlethread_delayed_workitem(
		ql_vmk_singlethread_delayed_workitem_t *d_work)
{
	ql_vmk_init_singlethread_workitem(&d_work->work, NULL, NULL, NULL);
	d_work->wq = NULL;
}


VMK_ReturnStatus ql_vmk_queue_delayed_work(vmk_TimerQueue tq,
    ql_vmk_singlethread_workqueue_t *wq,
    ql_vmk_singlethread_delayed_workitem_t *d_work, void *handler,
    void *data1, void *data2, unsigned long delay_msec)
{
	VMK_ReturnStatus rc;

	vmk_SpinlockLock(wq->lock);
	if (d_work->work.workitem_state == ACTIVE) {
		vmk_LogMessage("qfle3f: work %p timer already active", d_work);
		vmk_SpinlockUnlock(wq->lock);
		return VMK_FAILURE;
	}

	if (d_work->work.workitem_state == DELAY_ACTIVE)  {
		vmk_LogMessage("qfle3f: delayed work %p timer already active", d_work);
		rc = vmk_TimerCancel(d_work->timer, VMK_FALSE);
        if(rc != VMK_OK) {
            vmk_LogMessage("%s:%d :: Timer Cancel Failed", __func__, __LINE__);
        }
	}

	ql_vmk_init_singlethread_workitem(&d_work->work, handler, data1, data2);
	d_work->wq = wq;

	d_work->work.workitem_state = DELAY_ACTIVE;
	rc = vmk_TimerSchedule(tq,
		(void *)ql_vmk_handle_tmo,
		(void *)d_work,
		delay_msec * VMK_USEC_PER_MSEC,
		VMK_TIMER_DEFAULT_TOLERANCE,
		VMK_TIMER_ATTR_NONE,
		VMK_LOCKDOMAIN_INVALID,
		VMK_SPINLOCK_UNRANKED,
		&d_work->timer);

	vmk_SpinlockUnlock(wq->lock);
	if (rc != VMK_OK) {
		vmk_LogMessage("qfle3f: could not sched timer.\n");
	}
	return rc;
}

void ql_vmk_handle_tmo(ql_vmk_singlethread_delayed_workitem_t *d_work)
{
	if (!d_work) {
		vmk_LogMessage("d_work NULL!\n");
		return;
	}

	ql_vmk_singlethread_queue_work(d_work->wq, &d_work->work);
}

VMK_ReturnStatus ql_vmk_cancel_delayed_work(ql_vmk_singlethread_delayed_workitem_t *d_work)
{
    VMK_ReturnStatus status;
	status = vmk_TimerCancel(d_work->timer, VMK_FALSE);

    if(status == VMK_OK) {
        vmk_SpinlockLock(d_work->wq->lock);
	    d_work->work.workitem_state = INACTIVE;
        vmk_SpinlockUnlock(d_work->wq->lock);
    }

    return status;
}

VMK_ReturnStatus ql_vmk_cancel_delayed_work_sync(ql_vmk_singlethread_delayed_workitem_t *d_work)
{
    VMK_ReturnStatus status;
	status = vmk_TimerCancel(d_work->timer, VMK_TRUE);

    if(status == VMK_OK) {
        vmk_SpinlockLock(d_work->wq->lock);
	    d_work->work.workitem_state = INACTIVE;
        vmk_SpinlockUnlock(d_work->wq->lock);
    }
    return status;
}


inline void qfle3f_int_to_scsilun_with_sec_lun_id(vmk_uint16 lun,
							vmk_uint8 *scsi_lun,
							vmk_uint64 sllid)
{
	if (sllid != VMK_SCSI_INVALID_SECONDLEVEL_ID) {
		vmk_Memset(scsi_lun, 0, 8);
		scsi_lun[0] = (lun >> 8) & 0xFF;
		scsi_lun[1] = lun & 0xFF;
		scsi_lun[2] = (vmk_uint8)((sllid >> 56) & 0xFF); /* sllid msb */
		scsi_lun[3] = (vmk_uint8)((sllid >> 48) & 0xFF);
		scsi_lun[4] = (vmk_uint8)((sllid >> 40) & 0xFF);
		scsi_lun[5] = (vmk_uint8)((sllid >> 32) & 0xFF);
		scsi_lun[6] = (vmk_uint8)((sllid >> 24) & 0xFF);
		scsi_lun[7] = (vmk_uint8)((sllid >> 16) & 0xFF); /* sllid lsb */
	}
	else {
		qfle3f_u64ToCharArray(lun, sizeof(lun), scsi_lun);
	}
}

/*
 * Function Name : qfle3f_charArrayToVmku64
 *
 * DESCRIPTION:
 *  Convert a character array to a uint64 number
 *
 * PARAMETERS:
 *  input[in]:      The character array
 *  inputSize[in]:  Size of Character array
 *  output[out]:    The output uint64 number
 *
 * RETURN:
 *  None.
 *
 * NOTE:
 *  Most common usecase for this is for inputsize of 8.
 */
inline void qfle3f_charArrayToVmku64(vmk_uint8 *input,
		int inputSize, vmk_uint64* output)
{
	vmk_uint64 interOutput = 0;
	int i = 0;

	if(inputSize > WWN_SIZE) {
		*output = 0;
		return;
	}

	for(i = 0; i < inputSize; i++) {
		interOutput |= ((vmk_uint64)input[i] <<
				((inputSize-i-1)*(VMK_BITS_PER_BYTE)));
	}

	*output = interOutput;
}

/*
 * Function Name : qfle3f_u64ToCharArray_BE
 *
 * DESCRIPTION:
 *  Convert a character array to a uint64 number
 *
 * PARAMETERS:
 *  input[in]:      The integer value
 *  inputSize[in]:  Size of Character array
 *  output[out]:    The output char array
 *
 * RETURN:
 *  None.
 *
 * NOTE:
 *  Most common usecase for this is for inputsize of 8.
 */
void qfle3f_u64ToCharArray_BE(vmk_uint64 input,
		int inputSize, vmk_uint8 *output)
{
	int i = 0;

	vmk_Memset(output, 0, inputSize);

	for(i = 0; i < inputSize; i++) {
		output[i] = (input >>
				((inputSize-i-1)*(VMK_BITS_PER_BYTE))) & 0xFF;
	}
}

/*
 * Function Name : qfle3f_u64ToCharArray
 *
 * DESCRIPTION:
 *  Convert a character array to a uint64 number
 *
 * PARAMETERS:
 *  input[in]:      The integer value
 *  inputSize[in]:  Size of Character array
 *  output[out]:    The output char array
 *
 * RETURN:
 *  None.
 *
 * NOTE:
 *  Most common usecase for this is for inputsize of 8.
 */
void qfle3f_u64ToCharArray(vmk_uint64 input,
		int inputSize, vmk_uint8 *output) {
	vmk_uint8 counter = 0;
	vmk_uint16 numberToConsider;

	vmk_Memset(output, 0, inputSize);

	while(counter < inputSize) {
		numberToConsider = (vmk_uint16) (input >> (counter*VMK_BITS_PER_BYTE));
		output[counter] = (numberToConsider >> VMK_BITS_PER_BYTE) & 0xFF;
		output[counter+1] = numberToConsider & 0xFF;
		counter += 2;
	}
}


/**
 * Function Name : qfle3f_sleepWithCondition
 *
 * DESCRIPTION:
 *  Current World goes to sleep until the timeout happens or the
 *  condition is met.
 *
 *  "condition is met" is considered as function condition(void *data1)
 *  returning VMK_TRUE.
 *
 *  This function is needed because of the limitation of the vmk_WorldWait,
 *  refer to the vmkAPI doc.
 *
 * PARAMETERS:
 *  hba:            Global structure associated with a port.
 *  data:           The pointer that would be passed to the condition()
 *  event:          event on which the world should be sleeping on.
 *  lockHeld:       spinlock held while calling this function
 *  timeout:        The timeout, in msec, to sleep.
 *  description:    A String describing the event.
 *  checkCondition: The callback function for checking the condtion.
 *
 * RETURN:
 *  VMK_OK:     On success.
 */
VMK_ReturnStatus qfle3f_sleepWithCondition(struct qfle3fHBA *hba, void *data,
                        vmk_WorldEventID event, vmk_Lock lockHeld, vmk_uint32 timeout,
                        char *desciption,
                        vmk_Bool (*checkCondition)(void *data))
{
    vmk_TimerCycles expiryCycles = 0, currCycles;
    VMK_ReturnStatus status = VMK_TIMEOUT;

	if(timeout != VMK_TIMEOUT_UNLIMITED_MS)
	    expiryCycles = vmk_GetTimerCycles() + (timeout*vmk_TimerCyclesPerSecond());

    qfle3f_log(hba, LOG_SESS, "Going to sleep for %s : 0x%lx.",
                            desciption, event);

    currCycles = vmk_GetTimerCycles();

    if(checkCondition(data)) {
        qfle3f_log(hba, LOG_SESS, "No need to sleep, condition already met");
        return VMK_OK;
    }

    while(expiryCycles > currCycles || timeout==VMK_TIMEOUT_UNLIMITED_MS) {
		if(timeout != VMK_TIMEOUT_UNLIMITED_MS)
			timeout = (expiryCycles - currCycles)/(vmk_TimerCyclesPerSecond()/1000);

        status = vmk_WorldWait(event, lockHeld, timeout, desciption);
        qfle3f_err(hba, "Wokeup with status = %s", vmk_StatusToString(status));

        /* If status in VMK_BAD_PARAM, we are still holding the lock */
        if(status != VMK_BAD_PARAM) {
            if(lockHeld != VMK_LOCK_INVALID)
                vmk_SpinlockLock(lockHeld);
        }

        if(status == VMK_DEATH_PENDING) // TODO: Can this really happen ?
            return status;

        if(checkCondition(data)) {
            status = VMK_OK;
            break;
        }

        currCycles = vmk_GetTimerCycles();

        /* Reset the status to VMK_FAILURE */
        status = VMK_TIMEOUT;
    }

    qfle3f_err(hba, "Returning %s", vmk_StatusToString(status));
    return status;
}

