/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    http://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */
package org.apache.directory.studio.schemaeditor.controller.actions;


import java.util.Iterator;

import org.apache.directory.studio.connection.core.ConnectionCoreConstants;
import org.apache.directory.studio.schemaeditor.Activator;
import org.apache.directory.studio.schemaeditor.PluginConstants;
import org.apache.directory.studio.schemaeditor.controller.ProjectsHandler;
import org.apache.directory.studio.schemaeditor.model.Project;
import org.apache.directory.studio.schemaeditor.model.Project.ProjectState;
import org.apache.directory.studio.schemaeditor.view.wrappers.ProjectWrapper;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.IWorkbenchWindowActionDelegate;


/**
 * This action deletes one or more Projects from the ProjectsView.
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class DeleteProjectAction extends Action implements IWorkbenchWindowActionDelegate
{
    /** The associated viewer */
    private TableViewer viewer;


    /**
     * Creates a new instance of DeleteProjectAction.
     *
     * @param view
     *      the associated view
     */
    public DeleteProjectAction( TableViewer viewer )
    {
        super( Messages.getString( "DeleteProjectAction.DeleteProjectAction" ) ); //$NON-NLS-1$
        setToolTipText( Messages.getString( "DeleteProjectAction.DeleteProjectToolTip" ) ); //$NON-NLS-1$
        setId( PluginConstants.CMD_DELETE_PROJECT );
        setActionDefinitionId( PluginConstants.CMD_DELETE_PROJECT );
        setImageDescriptor( Activator.getDefault().getImageDescriptor( PluginConstants.IMG_DELETE ) );
        setEnabled( false );
        this.viewer = viewer;
        this.viewer.addSelectionChangedListener( new ISelectionChangedListener()
        {
            public void selectionChanged( SelectionChangedEvent event )
            {
                StructuredSelection selection = ( StructuredSelection ) event.getSelection();
                if ( selection.size() == 1 )
                {
                    setText( Messages.getString( "DeleteProjectAction.DeleteProjectAction" ) ); //$NON-NLS-1$
                    setEnabled( true );
                }
                else if ( selection.size() > 1 )
                {
                    setText( Messages.getString( "DeleteProjectAction.DeleteProjectsAction" ) ); //$NON-NLS-1$
                    setEnabled( true );
                }
                else
                {
                    setText( Messages.getString( "DeleteProjectAction.DeleteProjectAction" ) ); //$NON-NLS-1$
                    setEnabled( false );
                }
            }
        } );
    }


    /**
     * {@inheritDoc}
     */
    public void run()
    {
        ProjectsHandler projectsHandler = Activator.getDefault().getProjectsHandler();
        StructuredSelection selection = ( StructuredSelection ) viewer.getSelection();

        if ( !selection.isEmpty() )
        {
            StringBuilder title = new StringBuilder();
            StringBuilder message = new StringBuilder();

            int count = selection.size();

            if ( count <= 5 )
            {
                if ( count == 1 )
                {
                    // Only 1 project to delete
                    title.append( Messages.getString( "DeleteProjectAction.DeleteProjectTitle" ) ); //$NON-NLS-1$
                    message.append( Messages.getString( "DeleteProjectAction.SureDeleteFollowingProject" ) ); //$NON-NLS-1$
                }
                else
                {
                    // Between 2 to 5 projects to delete
                    title.append( Messages.getString( "DeleteProjectAction.DeleteProjectsTitle" ) ); //$NON-NLS-1$
                    message.append( Messages.getString( "DeleteProjectAction.SureDeleteFollowingProjects" ) ); //$NON-NLS-1$
                }

                // Appending the projects names
                for ( Iterator<?> iterator = selection.iterator(); iterator.hasNext(); )
                {
                    message.append( ConnectionCoreConstants.LINE_SEPARATOR );
                    message.append( "  - " ); //$NON-NLS-1$
                    message.append( ( ( ProjectWrapper ) iterator.next() ).getProject().getName() );
                }
            }
            else
            {
                // More than 5 projects to delete
                title.append( Messages.getString( "DeleteProjectAction.DeleteProjectsTitle" ) ); //$NON-NLS-1$
                message.append( Messages.getString( "DeleteProjectAction.SureDeleteSelectedProjects" ) ); //$NON-NLS-1$
            }

            // Showing the confirmation window
            if ( MessageDialog.openConfirm( viewer.getControl().getShell(), title.toString(), message.toString() ) )
            {
                for ( Iterator<?> iterator = selection.iterator(); iterator.hasNext(); )
                {
                    ProjectWrapper wrapper = ( ProjectWrapper ) iterator.next();
                    Project project = wrapper.getProject();

                    if ( project.getState() == ProjectState.OPEN )
                    {
                        // Closing the project before removing it. 
                        projectsHandler.closeProject( project );
                    }

                    projectsHandler.removeProject( project );
                }
            }
        }
    }


    /**
     * {@inheritDoc}
     */
    public void run( IAction action )
    {
        run();
    }


    /**
     * {@inheritDoc}
     */
    public void dispose()
    {
        // Nothing to do
    }


    /**
     * {@inheritDoc}
     */
    public void init( IWorkbenchWindow window )
    {
        // Nothing to do
    }


    /**
     * {@inheritDoc}
     */
    public void selectionChanged( IAction action, ISelection selection )
    {
        // Nothing to do
    }
}
