# The Development Cycle

## Prerequisites
1. [Development Environment](development-environment.md)   
2. [Toolchain Guide](toolchain-guide.md)

## Preface 
When you have developed and possibly manually tested the code you want to contribute, you should ensure that everything builds correctly. Commit your changes locally and perform the following steps, for each step the commands for both local and virtualized toolchain are listed. 

### Docker priviliges for virtualized toolchain users
[These instructions](toolchain-guide.md#docker-privileges) are relevant for most of the steps below 

### Using Podman for virtualized toolchain users
[These instructions](toolchain-guide.md#using-podman) are relevant for most of the steps below 

## Development cycle steps
### Set kubectl context to argocd namespace

Setting kubectl config to the argocd namespace is required for these steps to succeed.
All following commands in this guide assume the namespace is already set.

```shell
kubectl config set-context --current --namespace=argocd
```

### Pull in all build dependencies

As build dependencies change over time, you have to synchronize your development environment with the current specification. In order to pull in all required dependencies, issue:

* `make dep-ui` or `make dep-ui-local`

Argo CD recently migrated to Go modules. Usually, dependencies will be downloaded at build time, but the Makefile provides two targets to download and vendor all dependencies:

* `make mod-download` or `make mod-download-local` will download all required Go modules and
* `make mod-vendor` or `make mod-vendor-local` will vendor those dependencies into the Argo CD source tree

### Generate API glue code and other assets

Argo CD relies on Google's [Protocol Buffers](https://developers.google.com/protocol-buffers) for its API, and this makes heavy use of auto-generated glue code and stubs. Whenever you touched parts of the API code, you must re-generate the auto generated code.

* Run `make codegen` or `make codegen-local`, this might take a while
* Check if something has changed by running `git status` or `git diff`
* Commit any possible changes to your local Git branch, an appropriate commit message would be `Changes from codegen`, for example.

!!!note
    There are a few non-obvious assets that are auto-generated. You should not change the autogenerated assets, as they will be overwritten by a subsequent run of `make codegen`. Instead, change their source files. Prominent examples of non-obvious auto-generated code are `swagger.json` or the installation manifest YAMLs.

### Build your code and run unit tests

After the code glue has been generated, your code should build and the unit tests should run without any errors. Execute the following statements:

* `make build` or `make build-local`
* `make test` or `make test-local`

These steps are non-modifying, so there's no need to check for changes afterward.

### Lint your code base

In order to keep a consistent code style in our source tree, your code must be well-formed in accordance to some widely accepted rules, which are applied by a Linter.

The Linter might make some automatic changes to your code, such as indentation fixes. Some other errors reported by the Linter have to be fixed manually.

* Run `make lint` or `make lint-local` and observe any errors reported by the Linter
* Fix any of the errors reported and commit to your local branch
* Finally, after the Linter reports no errors, run `git status` or `git diff` to check for any changes made automatically by Lint
* If there were automatic changes, commit them to your local branch

If you touched UI code, you should also run the Yarn linter on it:

* Run `make lint-ui` or `make lint-ui-local`
* Fix any of the errors reported by it

### Run end-to-end tests

The final step is running the End-to-End testsuite, which ensures that your Kubernetes dependencies are working properly. This will involve starting all the Argo CD components on your computer. The end-to-end tests consists of two parts: a server component, and a client component.

* First, start the End-to-End server: `make start-e2e` or `make start-e2e-local`. This will spawn a number of processes and services on your system.
* When all components have started, run `make test-e2e` or `make test-e2e-local` to run the end-to-end tests against your local services.

To run a single test with a local toolchain, you can use `TEST_FLAGS="-run TestName" make test-e2e-local`.

For more information about End-to-End tests, refer to the [End-to-End test documentation](test-e2e.md).

## Common Make Targets

Here are some frequently used make targets (all will run on your machine):

### Local Toolchain Make Targets

* `make install-tools-local` - Install testing and building tools for the local toolchain 
* `make build-local` - Build Argo CD binaries
* `make test-local` - Run unit tests
* `make codegen-local` - Re-generate auto generated Swagger and Protobuf (after changing API code)
* `make lint-local` - Run linting
* `make pre-commit-local` - Run pre-commit checks
* `make start-e2e-local` - Start server for end-to-end tests
* `make test-e2e-local` - Run end-to-end tests
* `make serve-docs-local` - Serve documentation
* `make start-local` - Start Argo CD
* `make cli-local` - Build Argo CD CLI binary

### Virtualized Toolchain Make Targets

* `make verify-kube-connect` - Test whether the virtualized toolchain has access to your K8s cluster
* `make test-tools-image` - Prepare the environment of the virtualized chain
* `make build` - Build Argo CD binaries
* `make test` - Run unit tests
* `make codegen` - Re-generate auto generated Swagger and Protobuf (after changing API code)
* `make lint` - Run linting
* `make pre-commit` - Run pre-commit checks
* `make start-e2e` - Start server for end-to-end tests
* `make test-e2e` - Run end-to-end tests
* `make serve-docs` - Serve documentation
* `make start` - Start Argo CD

---
Congrats on making it to the end of this runbook! 🚀

For more on Argo CD, find us in Slack - <https://slack.cncf.io/> [#argo-contributors](https://cloud-native.slack.com/archives/C020XM04CUW)
